# -*- encoding: utf-8 -*-

from __future__ import print_function

import argparse
import errno
import os
from datetime import datetime

import kobo.conf
from productmd.composeinfo import ComposeInfo


class MissingConfigOptionError(RuntimeError):
    pass


def get_compose_dir(topdir, conf, compose_type="production", compose_date=None, compose_respin=None, compose_label=None):
    ci = ComposeInfo()
    try:
        ci.release.name = conf["release_name"]
        ci.release.short = conf["release_short"]
        ci.release.version = conf["release_version"]
        ci.release.is_layered = bool(conf.get("release_is_layered", False))
        ci.release.type = conf.get("release_type", "ga").lower()
        ci.release.internal = bool(conf.get("release_internal", False))

        if ci.release.is_layered:
            ci.base_product.name = conf["base_product_name"]
            ci.base_product.short = conf["base_product_short"]
            ci.base_product.version = conf["base_product_version"]
            ci.base_product.type = conf.get("base_product_type", "ga").lower()
    except KeyError as ex:
        raise MissingConfigOptionError("Option %s is missed from config" % str(ex))

    ci.compose.label = compose_label
    ci.compose.type = compose_type
    ci.compose.date = compose_date or datetime.today().strftime("%Y%m%d")
    ci.compose.respin = compose_respin or 0

    while 1:
        ci.compose.id = ci.create_compose_id()

        compose_dir = os.path.join(topdir, ci.compose.id)

        exists = False
        try:
            os.makedirs(compose_dir)
        except OSError as ex:
            if ex.errno == errno.EEXIST:
                exists = True
            else:
                raise

        if exists:
            ci.compose.respin += 1
            continue
        break

    open(os.path.join(compose_dir, "COMPOSE_ID"), "w").write(ci.compose.id)
    work_dir = os.path.join(compose_dir, "work", "global")
    os.makedirs(work_dir)
    ci.dump(os.path.join(work_dir, "composeinfo-base.json"))

    return compose_dir


def main(args):

    conf = kobo.conf.PyConfigParser()
    conf.load_from_file(args.config)
    next_dir = get_compose_dir(args.target_dir, conf, compose_type=args.type, compose_label=args.label)
    print(next_dir)


def run(arguments=None):
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--config',
        metavar='CONFIG',
        required=True,
        help='config file',
    )
    parser.add_argument(
        '--target-dir',
        metavar='PATH',
        required=True,
        help='create compose dir under this directory',
    )
    parser.add_argument(
        '--type',
        metavar='TYPE',
        choices=('production', 'test', 'nightly', 'ci'),
        default='production',
        help='compose type',
    )
    parser.add_argument(
        "--label",
        metavar="LABEL",
        help="specify compose label (example: Snapshot-1.0); required for production composes"
    )
    parser.add_argument(
        "--no-label",
        action="store_true",
        default=False,
        help="make a production compose without label"
    )

    args = parser.parse_args(arguments)
    if args.type == "production" and not args.label and not args.no_label:
        raise RuntimeError("label is required for a production compose, "
                           "or you can run this with '--no-label'")
    main(args)
