# -*- encoding: utf-8 -*-

from __future__ import print_function

import os
import argparse
import datetime
import fnmatch

import productmd.compose


class BadUsage(RuntimeError):
    pass


def version_sort(ver):
    return ver.split('.')


def compose_sort(compose):
    return (compose.info.compose.date, compose.info.compose.respin)


def _release_matches(release, name, short, version):
    short_match = not short or release.short == short
    name_match = not name or release.name == name
    version_match = not version or release.version == version
    return short_match and name_match and version_match


def _has_status(fullpath, status):
    with open(os.path.join(fullpath, 'STATUS')) as f:
        actual = f.read().strip()
        return fnmatch.fnmatchcase(actual, status)


def load_composes(path, name=None, short=None, version=None, ctype=None, status=None):
    composes = {}
    for item in os.listdir(path):
        fullpath = os.path.join(path, item)
        if not os.path.isdir(fullpath) or os.path.islink(fullpath):
            continue

        try:
            if status and not _has_status(fullpath, status):
                continue
        except IOError:
            # No STATUS file
            continue

        c = productmd.compose.Compose(fullpath)
        try:
            if not _release_matches(c.info.release, name, short, version):
                # Not interesting release, skip
                continue
            if ctype and c.info.compose.type != ctype:
                # Not correct compose type, skip
                continue
            composes.setdefault(c.info.release.name, {}).setdefault(c.info.release.version, []).append(c)
        except AttributeError:
            # Productmd failed to find the metadata, most likely not a compose directory.
            pass
    return composes


def sorted_ids(composes):
    """Sort composes by their release name, release version, compose date and
    respin, and return the list of IDs.
    """
    result = []
    for release in sorted(composes):
        for version in sorted(composes[release], key=version_sort):
            for compose in sorted(composes[release][version], key=compose_sort):
                result.append(compose.info.compose.id)
    return result


def print_sorted(composes):
    """Print all sorted compose IDs. """
    for compose_id in sorted_ids(composes):
        print(compose_id)


def check_single_line(composes):
    """Check that all composes have the same release name and version.

    Returns a list with the sorted composes.
    """
    if len(composes) == 1:
        versions = composes[composes.keys()[0]]
        if len(versions) == 1:
            return sorted(versions[versions.keys()[0]], key=compose_sort)
    raise BadUsage(
        'Expected to find composes for only one release.\n'
        'Consider adding some filters.'
    )


def find_preceding(composes, precedes):
    last = None
    for compose in composes:
        if compose.info.compose.id == precedes:
            break
        last = compose
    if not last:
        raise BadUsage('No preceding compose for {}.'.format(precedes))
    return last


def increment_compose_id(composeinfo):
    today = datetime.datetime.now().strftime('%Y%m%d')
    if composeinfo.compose.date == today:
        composeinfo.compose.respin += 1
    else:
        composeinfo.compose.date = today
        composeinfo.compose.respin = 0
    return composeinfo.create_compose_id()


def main(args):
    composes = load_composes(args.DIR, name=args.name, short=args.short,
                             version=args.ver, ctype=args.type, status=args.status)
    if not composes:
        raise BadUsage('No composes found')

    if args.next:
        composes = check_single_line(composes)
        print(increment_compose_id(composes[-1].info))
    elif args.last:
        composes = check_single_line(composes)
        print(composes[-1].info.compose.id)
    elif args.precedes:
        composes = check_single_line(composes)
        preceding = find_preceding(composes, args.precedes)
        print(preceding.info.compose.id)
    else:
        print_sorted(composes)


def run(arguments=None):
    parser = argparse.ArgumentParser()
    parser.add_argument('DIR', help='directory with composes')
    parser.add_argument('--short', metavar='SHORT',
                        help='only work with composes from release with this short name')
    parser.add_argument('--name', metavar='NAME',
                        help='only work with composes from release with this name')
    parser.add_argument('--ver', metavar='VERSION',
                        help='only work with composes from release with this version')
    parser.add_argument('--type', metavar='COMPOSE_TYPE',
                        help='only work with composes from release with this type')
    parser.add_argument('--status', metavar='STATUS', default='FINISHED*',
                        help='only work with composes with this status')
    g = parser.add_mutually_exclusive_group()
    g.add_argument('--precedes', metavar='COMPOSE_ID',
                   help='show compose preceding the given one')
    g.add_argument('--next', action='store_true',
                   help='show what next compose ID is going to be')
    g.add_argument('--last', action='store_true',
                   help='show most recent compose')

    args = parser.parse_args(arguments)
    main(args)
