# -*- encoding: utf-8 -*-

import unittest
import mock
import os
from StringIO import StringIO
import tempfile
import shutil

from compose_utils import copy_compose

from .helpers import get_compose, get_compose_path


class ChangelogTest(unittest.TestCase):

    def setUp(self):
        self.compose = get_compose('DP-1.0-20160315.t.0')
        self.dest = tempfile.mkdtemp(prefix='compose-copy-dest') + '/'

    def tearDown(self):
        shutil.rmtree(self.dest)

    def assertComposeId(self):
        with open(os.path.join(self.dest, 'COMPOSE_ID')) as f:
            self.assertEqual(f.read(), self.compose.info.compose.id)

    def assertNoComposeId(self):
        self.assertFalse(os.path.exists(os.path.join(self.dest, 'COMPOSE_ID')))

    @mock.patch('kobo.shortcuts.run')
    def test_copy_full(self, mock_run):
        copy_compose(self.compose, self.dest, variants=[], arches=[])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client',
                        self.compose.compose_path + '/Server',
                        self.dest],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_full_variant_from_explicit_path(self, mock_run):
        self.compose = get_compose(get_compose_path('DP-1.0-20160315.t.0') + '/compose/')
        copy_compose(self.compose, self.dest, variants=['Server'], arches=[])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        os.path.join(self.compose.compose_path, 'Server'),
                        self.dest],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_to_missing_dir(self, mock_run):
        self.dest = os.path.join(self.dest, 'missing', 'directory') + '/'
        copy_compose(self.compose, self.dest, variants=[], arches=[])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client',
                        self.compose.compose_path + '/Server',
                        self.dest],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_full_with_extra_opts(self, mock_run):
        copy_compose(self.compose, self.dest, variants=[], arches=[], rsync_opts=['--exclude=repodata'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh', '--exclude=repodata',
                        self.compose.compose_path + '/Client',
                        self.compose.compose_path + '/Server',
                        self.dest],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_variant(self, mock_run):
        copy_compose(self.compose, self.dest, variants=['Client'], arches=[])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client', self.dest],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_single_variant(self, mock_run):
        # Only client has i386
        copy_compose(self.compose, self.dest, variants=[], arches=['i386'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/i386',
                        self.dest + 'Client'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_source_in_single_variant(self, mock_run):
        copy_compose(self.compose, self.dest, variants=['Client'], arches=['src'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/source',
                        self.dest + 'Client'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_both_variants(self, mock_run):
        # Both Client and Server have x86_64
        copy_compose(self.compose, self.dest, variants=[], arches=['x86_64'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/x86_64',
                        self.dest + 'Client'],
                       stdout=True),
             mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Server/x86_64',
                        self.dest + 'Server'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_one_variant_with_filter(self, mock_run):
        copy_compose(self.compose, self.dest, variants=['Client'], arches=['x86_64'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/x86_64',
                        self.dest + 'Client'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_source_in_both_variants(self, mock_run):
        copy_compose(self.compose, self.dest, variants=[], arches=['src'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/source',
                        self.dest + 'Client'],
                       stdout=True),
             mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Server/source',
                        self.dest + 'Server'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_source_in_one_variant_with_filter(self, mock_run):
        copy_compose(self.compose, self.dest, variants=['Client'], arches=['src'])
        self.assertItemsEqual(
            mock_run.mock_calls,
            [mock.call(['rsync', '-avHh',
                        self.compose.compose_path + '/Client/source',
                        self.dest + 'Client'],
                       stdout=True)])
        self.assertComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_full_dry_run(self, mock_run):
        with mock.patch('sys.stdout', new=StringIO()) as out:
            copy_compose(self.compose, self.dest, variants=[], arches=[], dry_run=True)

        self.assertEqual(mock_run.mock_calls, [])
        self.assertItemsEqual(
            out.getvalue().strip().split('\n'),
            [' '.join(['rsync', '-avHh',
                       self.compose.compose_path + '/Client',
                       self.compose.compose_path + '/Server',
                       self.dest])])
        self.assertNoComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_variant_dry_run(self, mock_run):
        with mock.patch('sys.stdout', new=StringIO()) as out:
            copy_compose(self.compose, self.dest, variants=['Client'], arches=[], dry_run=True)

        self.assertEqual(mock_run.mock_calls, [])
        self.assertItemsEqual(
            out.getvalue().strip().split('\n'),
            [' '.join(['rsync', '-avHh',
                       self.compose.compose_path + '/Client', self.dest])])
        self.assertNoComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_single_variant_dry_run(self, mock_run):
        # Only client has i386
        with mock.patch('sys.stdout', new=StringIO()) as out:
            copy_compose(self.compose, self.dest, variants=[], arches=['i386'], dry_run=True)

        self.assertEqual(mock_run.mock_calls, [])
        self.assertItemsEqual(
            out.getvalue().strip().split('\n'),
            [' '.join(['rsync', '-avHh',
                       self.compose.compose_path + '/Client/i386',
                       self.dest + 'Client'])])
        self.assertNoComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_both_variants_dry_run(self, mock_run):
        with mock.patch('sys.stdout', new=StringIO()) as out:
            copy_compose(self.compose, self.dest, variants=[], arches=['x86_64'], dry_run=True)

        self.assertEqual(mock_run.mock_calls, [])
        self.assertItemsEqual(
            out.getvalue().strip().split('\n'),
            [' '.join(['rsync', '-avHh',
                       self.compose.compose_path + '/Client/x86_64',
                       self.dest + 'Client']),
             ' '.join(['rsync', '-avHh',
                       self.compose.compose_path + '/Server/x86_64',
                       self.dest + 'Server'])])
        self.assertNoComposeId()

    @mock.patch('kobo.shortcuts.run')
    def test_copy_one_arch_in_one_variant_with_filter_dry_run(self, mock_run):
        with mock.patch('sys.stdout', new=StringIO()) as out:
            copy_compose(self.compose, self.dest, variants=['Client'], arches=['x86_64'], dry_run=True)

        self.assertEqual(mock_run.mock_calls, [])
        self.assertEqual(
            out.getvalue().strip(),
            ' '.join(['rsync', '-avHh',
                      self.compose.compose_path + '/Client/x86_64',
                      self.dest + 'Client']))
        self.assertNoComposeId()


if __name__ == '__main__':
    unittest.main()
