# -*- coding: utf-8 -*-

import os
import shutil
import tempfile
import unittest
import mock
from StringIO import StringIO

from compose_utils import symlink

from . import helpers


class TestLatestSymlink(unittest.TestCase):
    def setUp(self):
        self.compose_dir = tempfile.mkdtemp()
        self.compose = helpers.get_compose('DP-1.0-20160315.t.1')
        self.compose_path = os.path.join(self.compose_dir, 'DP-1.0-20160315.t.1')

    def tearDown(self):
        shutil.rmtree(self.compose_dir)

    def _write_compose(self, is_layered=False):
        if is_layered:
            self.compose.info.release.is_layered = True
            self.compose.info.base_product.name = 'Base Product'
            self.compose.info.base_product.type = 'ga'
            self.compose.info.base_product.short = 'BP'
            self.compose.info.base_product.version = '15'

        path = os.path.join(self.compose_dir, self.compose.info.compose.id)
        metadata_dir = os.path.join(path, 'compose', 'metadata')
        os.makedirs(metadata_dir)
        self.compose.info.dump(os.path.join(metadata_dir, 'composeinfo.json'))
        return path

    def test_simple(self):
        path = self._write_compose()
        symlink.create_latest_link(path)
        self.assertEqual(os.readlink(os.path.join(self.compose_dir, 'latest-DP-1')),
                         'DP-1.0-20160315.t.1')

    def test_layered(self):
        path = self._write_compose(is_layered=True)
        symlink.create_latest_link(path)
        self.assertEqual(os.readlink(os.path.join(self.compose_dir, 'latest-DP-1-BP-15')),
                         'DP-1.0-20160315.t.1')

    def test_with_minor_version(self):
        path = self._write_compose()
        symlink.create_latest_link(path, minor_version=True)
        self.assertEqual(os.readlink(os.path.join(self.compose_dir, 'latest-DP-1.0')),
                         'DP-1.0-20160315.t.1')

    def test_layered_with_minor_version(self):
        path = self._write_compose(is_layered=True)
        symlink.create_latest_link(path, minor_version=True)
        self.assertEqual(os.readlink(os.path.join(self.compose_dir, 'latest-DP-1.0-BP-15')),
                         'DP-1.0-20160315.t.1')

    def test_remove_existing(self):
        os.symlink('foo', os.path.join(self.compose_dir, 'latest-DP-1'))

        path = self._write_compose()
        symlink.create_latest_link(path)
        self.assertEqual(os.readlink(os.path.join(self.compose_dir, 'latest-DP-1')),
                         'DP-1.0-20160315.t.1')

    def test_print_symlink(self):
        path = self._write_compose()
        with mock.patch('sys.stdout', new_callable=StringIO) as mock_out:
            symlink.print_latest_link(path)
        self.assertEqual(mock_out.getvalue(), 'latest-DP-1\n')

    def test_print_symlink_minor_version(self):
        path = self._write_compose()
        with mock.patch('sys.stdout', new_callable=StringIO) as mock_out:
            symlink.print_latest_link(path, minor_version=True)
        self.assertEqual(mock_out.getvalue(), 'latest-DP-1.0\n')
