# -*- encoding: utf-8 -*-

from __future__ import print_function

import errno
import os
from kobo import shortcuts


def copy_compose(compose, dest, arches=None, variants=None, dry_run=False, rsync_opts=[]):
    """Copy (possibly part of) a compose to another location.

    :param compose: a compose instance to work with
    :param dest: a file path where the to copy
    :param arches: a list of arches that should be copied; if not specified,
                   all arches are considered
    :param variants: a list of variant UIDs to copy; if not specified, all of
                     them are copied
    :param dry_run: When `True`, the commands are printed, but not actually ran
    """
    arches = set(arches or [])

    paths = []

    for variant in compose.info.variants.get_variants(recursive=True):
        variant_arches = set(variant.arches) | set(['src'])
        if variants and variant.uid not in variants:
            continue

        if not arches or (variant_arches & arches) == variant_arches:
            paths.extend(_collect_variant(compose, variant, arch=None))
        else:
            for arch in variant_arches:
                if arch not in arches:
                    continue
                paths.extend(_collect_variant(compose, variant, arch=arch))

    compose_path = compose.compose_path.rstrip('/')
    paths = _reduce(compose, paths)
    for parent, children in paths.items():
        sources = sorted([os.path.join(parent, child) for child in children])
        destination = os.path.join(dest, parent.replace(compose_path, '').lstrip('/'))
        _run_rsync(sources, destination, dry_run=dry_run, opts=rsync_opts)

    if not dry_run:
        with open(os.path.join(dest, 'COMPOSE_ID'), 'w') as f:
            f.write(compose.info.compose.id)


def _run_rsync(sources, destination, dry_run=False, opts=[]):
    cmd = ['rsync', '-avHh'] + opts + sources + [destination]
    print(' '.join(cmd))
    if not dry_run:
        try:
            os.makedirs(destination)
        except OSError as exc:
            if exc.errno == errno.EEXIST:
                pass
        shortcuts.run(cmd, stdout=True)


def _collect_variant(compose, variant, arch=None):
    """Find all paths in a variant that should be copied.
    If arch is not specified, that means all paths, otherwise only paths
    defined for the arch will be copied.
    """
    sources = set()
    for path_type in variant.paths._fields:
        paths = getattr(variant.paths, path_type)
        if arch and path_type.startswith('source'):
            if arch == 'src':
                sources = sources | set(paths.values())
            continue
        if not arch:
            sources = sources | set(paths.values())
        elif arch in paths:
            sources.add(paths[arch])

    # Collect image paths: The images directory is not listed in
    # composeinfo.json, therefore we iterate over relevant part of images.json
    # to extract paths from there.
    images = []
    try:
        variant_images = compose.images.images[variant.uid]
        if arch:
            images = variant_images[arch]
        else:
            for vararch_images in variant_images.values():
                images.extend(vararch_images)
    except (KeyError, RuntimeError):
        # Ignore missing images metadata or no images in the variant/arch
        # combination.
        pass
    for image in images:
        if not arch or image.arch == arch:
            sources.add(os.path.dirname(image.path))

    return sources


def _reduce(compose, paths):
    """Given a list of paths, try to find a minimal set that can be rsync-ed to
    achive the same result.

    The main idea is that if we are supposed to copy everything in a particular
    directory, we can just simply copy the parent.
    """
    # Create a mapping from parent dirs to a set of children that should be
    # copied.
    sources = [os.path.split(os.path.join(compose.compose_path, x))
               for x in sorted(paths)]
    mapping = {}
    for dir, content in sources:
        mapping.setdefault(dir, set()).add(content)

    while True:
        has_change = False
        for dir in mapping:
            if set(os.listdir(dir)) <= mapping[dir]:
                # The directory has all children marked, so we can remove it.
                # Instead, we will add it as a child of its own parent.
                mapping.pop(dir)
                parent, name = os.path.split(dir)
                mapping.setdefault(parent, set()).add(name)
                has_change = True

                # If any other path starts with the removed directory, we can
                # remove it too.
                for other in list(mapping.keys()):
                    if (other.rstrip('/') + '/').startswith(dir + '/'):
                        mapping.pop(other)

                break
        if not has_change:
            break

    return mapping
