# -*- coding: utf-8 -*-

from six.moves import configparser


def write_legacy_composeinfo(ci, file):
    """Write a legacy INI-style composeinfo data into `file`."""
    conf = configparser.SafeConfigParser(dict_type=SortedDict)
    _serialize_compose(conf, ci.compose)
    _serialize_product(conf, ci.release, ci.compose)
    if ci.release.is_layered:
        _serialize_base_product(conf, ci.base_product)
    _serialize_variants(conf, ci.variants)
    conf.write(file)


def _serialize_compose(conf, compose):
    conf.add_section('compose')
    conf.set('compose', 'date', compose.date)
    conf.set('compose', 'id', compose.id)
    conf.set('compose', 'respin', str(compose.respin))
    conf.set('compose', 'type', compose.type.lower())
    if compose.label:
        conf.set('compose', 'label', compose.label)


def _serialize_product(conf, release, compose=None, section='product'):
    conf.add_section(section)
    if compose:
        conf.set(section, 'family', release.name)
        conf.set(section, 'name', compose.id)
    else:
        conf.set(section, 'name', release.name)
    conf.set(section, 'short', release.short)
    conf.set(section, 'type', release.type.lower())
    conf.set(section, 'version', release.version)
    if release.is_layered:
        conf.set(section, 'is_layered', 'true')


def _serialize_base_product(conf, bp):
    conf.add_section('base_product')
    conf.set('base_product', 'name', bp.name)
    conf.set('base_product', 'short', bp.short)
    conf.set('base_product', 'version', bp.version)


def any_path(paths, options, arch):
    """Check if any path in `options` exists for `arch`."""
    for opt in options:
        if arch in getattr(paths, opt, {}):
            return True
    return False


def _serialize_paths(conf, section, paths, arch):
    conf.add_section(section)

    def _set(key, paths, arch):
        if arch in paths:
            conf.set(section, key, paths[arch])

    _set("os_tree", paths.os_tree, arch)
    _set("os_dir", paths.os_tree, arch)
    _set("packages", paths.packages, arch)
    _set("repository", paths.repository, arch)
    _set("isos", paths.isos, arch)
    _set("iso_dir", paths.isos, arch)
    _set("jigdos", paths.jigdos, arch)
    _set("jigdo_dir", paths.jigdos, arch)

    if any_path(paths, ["source_tree", "source_packages", "source_repository",
                        "source_isos", "source_jigdos"], arch):
        _set("source_tree", paths.source_tree, arch)
        _set("source_dir", paths.source_tree, arch)
        _set("source_packages", paths.source_packages, arch)
        _set("source_repository", paths.source_repository, arch)
        _set("source_isos", paths.source_isos, arch)
        _set("source_iso_dir", paths.source_isos, arch)
        _set("source_jigdos", paths.source_jigdos, arch)
        _set("source_jigdo_dir", paths.source_jigdos, arch)

    if any_path(paths, ["debug_tree", "debug_packages", "debug_repository"], arch):
        _set("debug_tree", paths.debug_tree, arch)
        _set("debug_dir", paths.debug_tree, arch)
        _set("debug_packages", paths.debug_packages, arch)
        _set("debug_repository", paths.debug_repository, arch)
        _set("debuginfo", paths.debug_repository, arch)


def _serialize_variant(conf, variant):
    section = 'variant-%s' % variant.uid
    conf.add_section(section)

    conf.set(section, 'id', variant.id)
    conf.set(section, 'uid', variant.uid)
    conf.set(section, 'name', variant.name)
    conf.set(section, 'type', variant.type)
    conf.set(section, 'arches', ','.join(sorted(variant.arches)))

    if variant.type == 'layered-product':
        _serialize_product(conf, variant.release, section='product-%s' % variant.uid)

    for arch in sorted(variant.arches):
        path_section = 'variant-%s.%s' % (variant.uid, arch)
        _serialize_paths(conf, path_section, variant.paths, arch)

    children = set()
    for child in variant.variants.values():
        _serialize_variant(conf, child)
        children.add(child.uid)

    if children:
        conf.set(section, 'variants', ','.join(sorted(children)))


def _serialize_variants(conf, variants):
    ids = sorted(variants.variants.keys())
    conf.set('product', 'variants', ','.join(ids))
    for variant_id in ids:
        variant = variants.variants[variant_id]
        _serialize_variant(conf, variant)


class SortedDict(dict):
    """A dictionary that always returns keys sorted."""
    def __iter__(self):
        for key in self.keys():
            yield key

    def iterkeys(self):
        for key in self.keys():
            yield key

    def itervalues(self):
        for key in self.keys():
            yield self[key]

    def keys(self):
        return sorted(dict.keys(self), reverse=False)

    def iteritems(self):
        for key in self.keys():
            yield (key, self[key])

    def items(self):
        return self.iteritems()
