# -*- coding: utf-8 -*-

import errno
import os
import productmd


class ComposeSymlink(object):
    def __init__(self, compose_path):
        self.compose = productmd.compose.Compose(compose_path)
        # The path points to the compose/ subdirectory.
        self.path = os.path.dirname(self.compose.compose_path)

    def _link(self, name):
        """Create a symbolic link with given name at the same location as the
        compose. Existing file at the given path will be deleted.
        """
        symlink = os.path.join(self.path, '..', name)
        try:
            os.unlink(symlink)
        except OSError as exc:
            if exc.errno != errno.ENOENT:
                raise

        os.symlink(os.path.basename(self.path), symlink)

    def get_latest_link_name(self, minor_version=False):
        """Get name of latest- symlink for current compose."""
        fmt = 'latest-{0.release.short}-{0.release.major_version}'
        if minor_version:
            fmt += '.{0.release.minor_version}'
        if self.compose.info.release.is_layered:
            fmt += '-{0.base_product.short}-{0.base_product.version}'
        return fmt.format(self.compose.info)

    def print_latest_link(self, minor_version=False):
        """Print a latest- symlink with optional minor version."""
        print(self.get_latest_link_name(minor_version=minor_version))

    def create_latest_link(self, minor_version=False):
        """Create a latest- symlink with optional minor version."""
        self._link(self.get_latest_link_name(minor_version=minor_version))


def print_latest_link(compose_path, minor_version=False):
    """Convenience function for easy printing of latest symlink."""
    ComposeSymlink(compose_path).print_latest_link(minor_version)


def create_latest_link(compose_path, minor_version=False):
    """Convenience function for easy creation of latest symlink."""
    ComposeSymlink(compose_path).create_latest_link(minor_version)
