# -*- encoding: utf-8 -*-

import json
import locale
import mock
try:
    import unittest2 as unittest
except ImportError:
    import unittest

from kobo.rpmlib import ChangelogEntry

from .helpers import get_compose, get_fixture

import productmd.compose
from compose_utils.changelog import ComposeChangelog, get_changelog_diff_from_headers


DUMMY_FIREFOX_SIZE = 5966 + 5934
OLD_DUMMY_FIREFOX_SIZE = 5878 + 5842
DUMMY_ELINKS_SIZE = 5850 + 5826 + 5818
DUMMY_TFTP_SIZE = 5842 + 5814 + 5806

DUMMY_FIREFOX = {
    'size': DUMMY_FIREFOX_SIZE,
    'name': 'Dummy-firefox',
    'old_nvr': 'Dummy-firefox-16.0.1-1',
    'old_rpms': ['Dummy-firefox'],
    'common_rpms': ['Dummy-firefox'],
    'rpms': ['Dummy-firefox'],
    'changelog': [u'* Tue Mar 15 2016 Lubomír Sedlář <lubomir.sedlar@gmail.com> - 1:0.1.0-1\n- new version'],
    'added_rpms': [],
    'dropped_rpms': [],
    'nvr': 'Dummy-firefox-1:0.1.0-1',
    'summary': 'A dummy firefox package',
    'size_change': DUMMY_FIREFOX_SIZE - OLD_DUMMY_FIREFOX_SIZE,
}

DUMMY_ELINKS = {
    'size': DUMMY_ELINKS_SIZE,
    'name': 'dummy-elinks',
    'rpms': ['dummy-elinks'],
    'nvr': 'dummy-elinks-2.6-2',
    'summary': 'A dummy elinks package',
}

DUMMY_TFTP = {
    'size': DUMMY_TFTP_SIZE,
    'name': 'dummy-tftp',
    'rpms': ['dummy-tftp'],
    'nvr': 'dummy-tftp-5.2-6',
    'summary': 'A dummy tftp package',
}


DUMMY_CLOUD_INIT = {
    'size': 5946,
    'name': 'cloud-init',
    'old_nvr': 'cloud-init-0.7.9-8.module_813d223c',
    'old_rpms': ['cloud-init'],
    'common_rpms': ['cloud-init'],
    'rpms': ['cloud-init'],
    'changelog': [u'* Tue Sep 05 2017 Lubomír Sedlář <lsedlar@redhat.com> - 0.7.9-9.module_f8c7dcdc\n- First release'],
    'added_rpms': [],
    'dropped_rpms': [],
    'nvr': 'cloud-init-0.7.9-9.module_f8c7dcdc',
    'summary': 'A dummy package',
    'size_change': 88,
}

DUMMY_NET_TOOLS = {
    'name': 'net-tools',
    'nvr': 'net-tools-2.0-0.44.20160912git.module_813d223c',
    'rpms': ['net-tools'],
    'size': 5910,
    'summary': 'A dummy package',
}


class ChangelogTest(unittest.TestCase):

    def setUp(self):
        locale.setlocale(locale.LC_TIME, 'C')

    def test_changelog(self):
        old_compose = get_compose('DP-1.0-20160315.t.0')
        new_compose = get_compose('DP-1.0-20160315.t.1')
        changelog = ComposeChangelog()

        self.maxDiff = None

        data = changelog.get_changelog(old_compose, new_compose)
        self.assertEqual(data['old_compose'], 'DP-1.0-20160315.t.0')
        self.assertEqual(data['new_compose'], 'DP-1.0-20160315.t.1')
        self.assertItemsEqual(data['added_packages'], [DUMMY_ELINKS])
        self.assertItemsEqual(data['dropped_packages'], [DUMMY_TFTP])
        self.assertItemsEqual(data['upgraded_packages'], [DUMMY_FIREFOX])
        self.assertItemsEqual(data['downgraded_packages'], [])
        self.assertDictEqual(data['added_images'][0], {
            "arch": "s390x",
            "bootable": False,
            "checksums": {
                "md5": "2d0a0e02d6a2225fba2c4cdf41d45ff7",
                "sha1": "8788b0aa4974440f7ca12c5dd0cc6dfc78314fdd",
                "sha256": "1b756cddb139e57fd6a07af72a7d52a9dc01c5fb0b9d37ef16fdf73a1a0a19af"
            },
            "disc_count": 1,
            "disc_number": 1,
            "format": "iso",
            "implant_md5": "0a2fc4f0e6f926c860e5df6c5ff0df32",
            "mtime": 1458031712,
            "path": "Server/s390x/iso/DP-1.0-20160315.t.1-Server-s390x-dvd1.iso",
            "size": 493568,
            "type": "dvd",
            "volume_id": "DP-1.0 Server.s390x",
            "subvariant": "Server"
        })
        self.assertDictEqual(data['dropped_images'][0], {
            "arch": "x86_64",
            "bootable": False,
            "checksums": {
                "md5": "7ea0d315e4b51006eb7c1de3da763c55",
                "sha1": "268bf671d5e62f35b70a2db5b1198b861d4b0eb8",
                "sha256": "32fedd333df5a0c78f3dd8c2634f8a4ab2328c7e0c9904dd06ac865fa723134f"
            },
            "disc_count": 1,
            "disc_number": 1,
            "format": "iso",
            "implant_md5": "128abd11392887a8abb2bbf119889791",
            "mtime": 1458031335,
            "path": "Client/x86_64/iso/DP-1.0-20160315.t.0-Client-x86_64-dvd1.iso",
            "size": 516096,
            "type": "dvd",
            "volume_id": "DP-1.0 Client.x86_64",
            "subvariant": "Client"
        })
        self.assertDictEqual(data['summary'], {
            'added_images': 1,
            'dropped_images': 1,
            'added_packages': 1,
            'added_packages_size': DUMMY_ELINKS_SIZE,
            'upgraded_packages': 1,
            'upgraded_packages_size': DUMMY_FIREFOX_SIZE,
            'upgraded_packages_size_change': DUMMY_FIREFOX_SIZE - OLD_DUMMY_FIREFOX_SIZE,
            'dropped_packages': 1,
            'dropped_packages_size': DUMMY_TFTP_SIZE,
            'downgraded_packages': 0,
            'downgraded_packages_size': 0,
            'downgraded_packages_size_change': 0,
        })

    def test_changelog_on_non_compose_dir_old(self):
        old_compose = productmd.compose.Compose('/tmp/not_here')
        new_compose = get_compose('DP-1.0-20160315.t.1')
        changelog = ComposeChangelog()

        with self.assertRaises(RuntimeError) as ctx:
            changelog.get_changelog(old_compose, new_compose)

        self.assertEqual('Failed to load metadata from /tmp/not_here',
                         str(ctx.exception))

    def test_changelog_on_non_compose_dir_new(self):
        old_compose = get_compose('DP-1.0-20160315.t.0')
        new_compose = productmd.compose.Compose('/tmp/not_here')
        changelog = ComposeChangelog()

        with self.assertRaises(RuntimeError) as ctx:
            changelog.get_changelog(old_compose, new_compose)

        self.assertEqual('Failed to load metadata from /tmp/not_here',
                         str(ctx.exception))

    def test_changelog_with_modular_suffix(self):
        old_compose = get_compose('Fedora-Modular-Bikeshed-20171004.n.0')
        new_compose = get_compose('Fedora-Modular-Bikeshed-20171005.n.0')
        changelog = ComposeChangelog(strip_suffix='module_')

        data = changelog.get_changelog(old_compose, new_compose)
        self.assertEqual(data['added_images'], [])
        self.assertEqual(data['dropped_images'], [])
        self.assertEqual(data['added_packages'], [])
        self.assertEqual(data['dropped_packages'], [DUMMY_NET_TOOLS])
        self.assertEqual(data['downgraded_packages'], [])
        self.assertEqual(data['upgraded_packages'], [DUMMY_CLOUD_INIT])


class TestFormat(unittest.TestCase):
    def setUp(self):
        super(TestFormat, self).setUp()
        with open(get_fixture('changelog-data.json')) as f:
            self.data = json.load(f)

    def test_summary(self):
        changelog = ComposeChangelog()
        with open(get_fixture('summary-full.txt')) as f:
            expected = f.read().split('\n')
        self.assertEqual(changelog._get_summary(self.data),
                         expected)

    def test_brief(self):
        changelog = ComposeChangelog()
        changelog._get_summary = mock.Mock(return_value=[])
        with open(get_fixture('brief-full.txt')) as f:
            expected = f.read().split('\n')
        self.assertEqual(changelog.get_brief_log(self.data).split('\n'),
                         expected)

    def test_verbose_full(self):
        changelog = ComposeChangelog()
        changelog._get_summary = mock.Mock(return_value=[])
        with open(get_fixture('verbose-full.txt')) as f:
            expected = f.read().split('\n')
        self.maxDiff = None
        self.assertEqual(changelog.get_verbose_log(self.data).split('\n'),
                         expected)

    def test_verbose_short(self):
        changelog = ComposeChangelog()
        changelog._get_summary = mock.Mock(return_value=[])
        with open(get_fixture('verbose-short.txt')) as f:
            expected = f.read().split('\n')
        self.maxDiff = None
        self.assertEqual(changelog.get_verbose_log(self.data, shorten=True).split('\n'),
                         expected)


class TestCompareChangelogs(unittest.TestCase):
    def test_select_new(self):
        c1 = ChangelogEntry(b'John Doe <jdoe@example.com> 1.0-1', 1000000, 'change 1')
        c2 = ChangelogEntry(b'John Doe <jdoe@example.com> 1.1-1', 2000000, 'change 2')
        old = mock.Mock(changelogs=[c1])
        new = mock.Mock(changelogs=[c2, c1])
        clog = get_changelog_diff_from_headers(old, new)
        self.assertEqual(clog, [c2])

    def test_select_with_same_date(self):
        c1 = ChangelogEntry(b'John Doe <jdoe@example.com> 1.0-1', 1000000, 'change 1')
        c2 = ChangelogEntry(b'John Doe <jdoe@example.com> 1.1-1', 1000000, 'change 2')
        old = mock.Mock(changelogs=[c1])
        new = mock.Mock(changelogs=[c2, c1])
        clog = get_changelog_diff_from_headers(old, new)
        self.assertEqual(clog, [c2])

    def test_no_subset(self):
        c1 = ChangelogEntry(b'John Doe <jdoe@example.com> 0:2.0-1', 1000000, 'change 1')
        c2 = ChangelogEntry(b'John Doe <jdoe@example.com> 1:1.1-1', 1000000, 'change 2')
        c3 = ChangelogEntry(b'John Doe <jdoe@example.com> 1:1.2-1', 1000000, 'change 3')
        c4 = ChangelogEntry(b'John Doe <jdoe@example.com> 1:1.3-1', 1000000, 'change 4')
        old = mock.Mock(changelogs=[c3, c1])
        new = mock.Mock(changelogs=[c4, c2, c1])
        clog = get_changelog_diff_from_headers(old, new)
        self.assertEqual(clog, [c4])


if __name__ == '__main__':
    unittest.main()
