# -*- encoding: utf-8 -*-

import json
import mock
import unittest
import shutil
import tempfile
import os

from freezegun import freeze_time
from six import StringIO

from compose_utils import create_next_dir


class CreateNextDirTest(unittest.TestCase):
    def setUp(self):
        self.topdir = tempfile.mkdtemp()
        self.target_dir = tempfile.mkdtemp(prefix='_compose_', dir=self.topdir)

    def tearDown(self):
        shutil.rmtree(self.topdir)

    def _write_config_from_string(self, string):
        config_file = os.path.join(self.topdir, 'fake.conf')
        lines = [l.strip() for l in string.split('\n') if l.strip() is not '']
        with open(config_file, 'w') as f:
            for line in lines:
                f.write('%s\n' % line)
        return config_file

    @freeze_time("2016-01-01")
    def test_create_next_production_compose_dir_with_label(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--label=Snapshot-2.0'])
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.0')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_create_next_production_compose_dir_with_no_label(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.0')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_create_next_production_compose_dir_error_without_label(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with self.assertRaises(RuntimeError) as ctx:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config])
        self.assertIn("label is required for a production compose, or you can run this with '--no-label'",
                      str(ctx.exception))

    @freeze_time("2016-01-01")
    def test_create_next_test_compose_dir(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--type=test'])
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.t.0')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_create_next_nightly_compose_dir(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--type=nightly'])
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.n.0')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_create_next_ci_compose_dir(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--type=ci'])
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.ci.0')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_compose_id_increment(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)

        os.makedirs(os.path.join(self.target_dir, 'DP-1.0-20160101.0'))
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])

        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.1')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

        os.makedirs(os.path.join(self.target_dir, 'DP-1.0-20160101.2'))
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])

        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.3')
        self.assertEqual(out.getvalue().strip(), compose_dir)
        self.assertTrue(os.path.isdir(compose_dir))

    @freeze_time("2016-01-01")
    def test_miss_option_in_config(self):
        cfg = """
            release_name = "Dummy Product"
            release_version = "1.0"
        """
        config = self._write_config_from_string(cfg)
        with self.assertRaises(create_next_dir.MissingConfigOptionError) as ctx:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])

        self.assertIn("Option 'release_short' is missed from config", str(ctx.exception))

    @freeze_time("2016-01-01")
    def test_no_permission_on_creating_compose_dir(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.0')
        config = self._write_config_from_string(cfg)
        os.chmod(self.target_dir, 555)
        with self.assertRaises(OSError) as ctx:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])
        self.assertIn("[Errno 13] Permission denied: '%s'" % compose_dir, str(ctx.exception))
        # restore write permission so it can be removed
        os.chmod(self.target_dir, 777)

    @freeze_time("2016-01-01")
    def test_create_compose_id(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.0')
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])

        self.assertEqual(out.getvalue().strip(), compose_dir)
        compose_id_file = os.path.join(compose_dir, 'COMPOSE_ID')
        self.assertTrue(os.path.exists(compose_id_file))
        with open(compose_id_file, 'r') as f:
            self.assertIn('DP-1.0-20160101.0', f.readlines())

    @freeze_time("2016-01-01")
    def test_create_base_composeinfo(self):
        cfg = """
            release_name = "Dummy Product"
            release_short = "DP"
            release_version = "1.0"
        """
        compose_dir = os.path.join(self.target_dir, 'DP-1.0-20160101.0')
        config = self._write_config_from_string(cfg)
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            create_next_dir.run(['--target-dir=%s' % self.target_dir,
                                 '--config=%s' % config,
                                 '--no-label'])
        self.assertEqual(out.getvalue().strip(), compose_dir)

        work_dir = os.path.join(compose_dir, "work", "global")
        base_composeinfo_file = os.path.join(work_dir, "composeinfo-base.json")
        self.assertTrue(os.path.exists(base_composeinfo_file))
        with open(base_composeinfo_file, 'r') as f:
            info = json.load(f)
            compose = info.get('payload', {}).get('compose', {})
            self.assertEqual(compose.get('id', ''), 'DP-1.0-20160101.0')
