# -*- encoding: utf-8 -*-

import mock
import unittest
import tempfile
import shutil
import os
import json
from six import StringIO
from freezegun import freeze_time

from compose_utils import list


def _make_composeinfo(compose_id, name, ver, ctype, date, respin, short):
    return {
        "header": {
            "version": "1.0"
        },
        "payload": {
            "compose": {
                "date": date,
                "id": compose_id,
                "respin": respin,
                "type": ctype
            },
            "release": {
                "name": name,
                "short": short,
                "version": ver
            },
            "variants": {}
        }
    }


def _write_status(path, status):
    with open(os.path.join(path, 'STATUS'), 'w') as f:
        f.write(status)


class ListTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()

    def tearDown(self):
        shutil.rmtree(self.dir)

    def _create(self, name, ver, ctype, date, respin, short='Test', status='FINISHED'):
        compose_id = '{short}-{ver}-{date}.{t}.{respin}'.format(
            short=short, ver=ver, date=date, t=ctype[0], respin=respin)
        dirname = os.path.join(self.dir, compose_id, 'compose', 'metadata')
        os.makedirs(dirname)
        with open(os.path.join(dirname, 'composeinfo.json'), 'w') as f:
            json.dump(_make_composeinfo(compose_id, name, ver, ctype, date, respin, short), f)
        _write_status(os.path.join(self.dir, compose_id), status)

    def test_list_all(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0)
        self._create('Test', '1.0', 'test', '20160316', 2)
        self._create('Test', '1.0', 'test', '20160316', 1)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160315.t.0',
                          'Test-1.0-20160316.t.0',
                          'Test-1.0-20160316.t.1',
                          'Test-1.0-20160316.t.2'])

    def test_list_skip_failed(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0, status='FINISHED_INCOMPLETE')
        self._create('Test', '1.0', 'test', '20160316', 2, status='DOOMED')
        self._create('Test', '1.0', 'test', '20160316', 1, status='DOOMED')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160315.t.0',
                          'Test-1.0-20160316.t.0'])

    def test_list_only_failed(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0, status='FINISHED_INCOMPLETE')
        self._create('Test', '1.0', 'test', '20160316', 2, status='DOOMED')
        self._create('Test', '1.0', 'test', '20160316', 1, status='DOOMED')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--status=DOOMED'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.1',
                          'Test-1.0-20160316.t.2'])

    def test_skip_non_dirs(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        with open(os.path.join(self.dir, 'some_file'), 'w') as f:
            f.write("Ceci n'est pas un compose")

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.0'])

    def test_skip_symlinks(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        os.symlink('Test-1.0-20160316.t.0', os.path.join(self.dir, 'latest'))

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.0'])

    def test_skip_non_compose_dirs(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        os.makedirs(os.path.join(self.dir, 'some-directory'))

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.0'])

    def test_list_filter(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Different Name', '1.0', 'test', '20160316', 0, short='DN')
        self._create('Test', '1.1', 'test', '20160316', 0)
        self._create('Test', '1.0', 'nightly', '20160316', 0)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--name=Test', '--ver=1.0', '--type=test'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.0'])

    @freeze_time("2016-03-16")
    def test_find_next_for_today(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0)
        self._create('Test', '1.0', 'test', '20160316', 2)
        self._create('Test', '1.0', 'test', '20160316', 1)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--next'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.3'])

    @freeze_time("2016-03-17")
    def test_find_next_first_today(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0)
        self._create('Test', '1.0', 'test', '20160316', 2)
        self._create('Test', '1.0', 'test', '20160316', 1)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--next'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160317.t.0'])

    def test_find_preceding(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0)
        self._create('Test', '1.0', 'test', '20160316', 2)
        self._create('Test', '1.0', 'test', '20160316', 1)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--precedes=Test-1.0-20160316.t.2'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.1'])

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--precedes=Test-1.0-20160316.t.0'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160315.t.0'])

    def test_find_last(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160315', 0)
        self._create('Test', '1.0', 'test', '20160316', 2)
        self._create('Test', '1.0', 'test', '20160316', 1)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--last'])

        self.assertEqual(out.getvalue().strip().split('\n'),
                         ['Test-1.0-20160316.t.2'])

    def test_find_last_on_empty_dir(self):
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--last'])

        self.assertEqual(out.getvalue(), '')

    def test_find_preceding_on_empty_dir(self):
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--precedes=Foo'])

        self.assertEqual(out.getvalue(), '')

    def test_find_next_on_empty_dir(self):
        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--next'])

        self.assertEqual(out.getvalue(), '')

    @freeze_time("2016-03-16")
    def test_find_next_with_multiple_releases(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Different Name', '1.0', 'test', '20160316', 0, short='DN')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--next'])

        self.assertEqual(out.getvalue(), '')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--next', '--name=Test'])

        self.assertEqual(out.getvalue(), 'Test-1.0-20160316.t.1\n')

    def test_find_last_with_multiple_releases(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Different Name', '1.0', 'test', '20160316', 0, short='DN')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--last'])

        self.assertEqual(out.getvalue(), '')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--last', '--name=Test'])

        self.assertEqual(out.getvalue(), 'Test-1.0-20160316.t.0\n')

    def test_find_preceding_with_multiple_releases(self):
        self._create('Test', '1.0', 'test', '20160316', 0)
        self._create('Test', '1.0', 'test', '20160316', 1)
        self._create('Different Name', '1.0', 'test', '20160316', 0, short='DN')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--precedes=Test-1.0-20160316.t.0'])

        self.assertEqual(out.getvalue(), '')

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            list.run([self.dir, '--precedes=Test-1.0-20160316.t.1', '--name=Test'])

        self.assertEqual(out.getvalue(), 'Test-1.0-20160316.t.0\n')

    def test_find_preceding_on_first(self):
        self._create('Test', '1.0', 'test', '20160316', 0)

        with mock.patch('sys.stdout', new_callable=StringIO) as out:
            with self.assertRaises(list.BadUsage):
                list.run([self.dir, '--precedes=Test-1.0-20160316.t.0'])

        self.assertEqual(out.getvalue(), '')
