#!/usr/bin/env python3

# Copyright Red Hat
#
# This file is part of wikitcms.
#
# wikitcms is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Adam Williamson <awilliam@redhat.com>

# this code is kinda irredeemable from these perspectives
# pylint: disable=too-many-locals,too-many-statements,too-many-branches

"""relval manipulates release validation events."""

import argparse
import json
import logging
import os
import shutil
import sys
import tempfile
from collections import defaultdict, OrderedDict

# for an exception; we'll need to change this when python-bugzilla
# ditches the xmlrpc API...
from xmlrpc.client import Fault

import bugzilla
import wikitcms.helpers as hl
import wikitcms.release as rl
import wikitcms.result as rs
import wikitcms.wiki as wk
from mwclient import errors as mwe

import relval.user_stats as uss
import relval.testcase_stats as tcs
import relval.report_results as rrs
import relval.size_check as rsc

logger = logging.getLogger(__name__)


## UTILITY FUNCTIONS ##


def comment_check(string):
    """Just a tiny wrapper to raise the right kind of exception."""
    try:
        return rrs.comment_string(string)
    except ValueError as err:
        raise argparse.ArgumentTypeError(err)


def guess_release(site):
    """Guess the release to operate on: uses the 'next' release so far
    as the wiki is concerned.
    """
    release = int(site.expandtemplates("{{FedoraVersionNumber|next}}"))
    print(f"Guessing release: {str(release)}")
    return release


def setup_site(args):
    """Access and, if login is True, log in to either the Fedora wiki
    (test=False) or the Fedora staging wiki (test=anything else, e.g.
    True). Returns a wikitcms Wiki instance.
    """
    if not args.test:
        site = wk.Wiki("fedoraproject.org")
    else:
        site = wk.Wiki("stg.fedoraproject.org")
    if args.login is True:
        site.login()
    return site


def parse_args():
    """Parse command-line arguments, set up the site instance, and
    call the appropriate sub-command method.
    """
    # Help texts shared by multiple sub-commands.
    release_help = "The Fedora release to operate on (22, 23, 24...)"
    milestone_help_specific = "The milestone to operate on (Alpha, Beta, Final, RC)"
    milestone_help = (
        "The milestone to operate on (Alpha, Beta, Final, RC, Branched, "
        "Rawhide). Must be specified for non-nightly; for a nightly event, "
        "will be guessed if not specified"
    )
    compose_help = "The compose to operate on (1.1, 20160314.n.0)"
    username_help = "DEPRECATED: your wiki (FAS) username. No longer used"
    force_help = "Force creation even if page(s) already exist"
    test_help = "Operate on the staging wiki (for testing)"
    filter_help = (
        "Keyword(s) to filter pages that will be processed. Can "
        "be specified more than once, all pages matching ANY "
        "specified keyword will be included"
    )

    # See the argparse instructions for all of this. Use of 'metavar'
    # avoids the entire ranges of valid values being shown in the help
    parser = argparse.ArgumentParser(
        description=(
            "Tool for various tasks related to Fedora release validation "
            "testing. 'compose' can create result pages for TC/RC and nightly "
            "validation test events. 'user-stats' and 'testcase-stats' can "
            "output statistics related to validation testing. 'report-results'"
            " lets you report test results. See each command's help for more "
            "details."
        ),
        epilog=(
            "For commands which require logging in to the wiki, on first use, "
            "a browser window will open and walk you through the Fedora auth "
            "process. An authentication token will be created to authenticate "
            "you for subsequent uses. After a while, the token will expire, "
            "and you will go through the browser authentication process again."
        ),
    )
    parser.add_argument(
        "-l",
        "--loglevel",
        help="The level of log messages to show",
        choices=("debug", "info", "warning", "error", "critical"),
        default="info",
    )
    # python 3 issue workaround, see:
    # https://stackoverflow.com/questions/23349349
    # http://bugs.python.org/issue16308
    subparsers = parser.add_subparsers(dest="subcommand")
    subparsers.required = True

    parser_compose = subparsers.add_parser(
        "compose",
        description="Create Fedora release validation event result "
        "pages. You must specify EITHER --release, --milestone and --compose "
        "OR --cid. You can create an event before the compose has completed, "
        "but you may not be able to use --cid, and the download page will "
        "not be created, you must remember to create that after the compose "
        "completes. This command requires login; see main help information "
        "about authentication.",
    )
    parser_compose.add_argument(
        "-r",
        "--release",
        help=release_help,
        type=int,
        metavar="24-99",
        choices=list(range(24, 100)),
    )
    parser_compose.add_argument(
        "-m",
        "--milestone",
        help=milestone_help,
        choices=["Beta", "Final", "RC", "Branched", "Rawhide"],
    )
    parser_compose.add_argument("-c", "--compose", help=compose_help, metavar="1.1 or 20160314.n.0")
    parser_compose.add_argument(
        "--modular",
        help="Operate on modular compose (deprecated, effect is same as --dist=Fedora-Modular)",
        action="store_true",
    )
    parser_compose.add_argument(
        "--dist",
        help="Dist (compose shortname, e.g. Fedora or Fedora-IoT) to operate on",
        default="Fedora",
    )
    parser_compose.add_argument(
        "-i",
        "--cid",
        help="Compose ID of a Pungi 4 compose to create a "
        "validation event for. If this is a 'production' compose - i.e. "
        "a candidate compose, not a nightly compose - it must be "
        "completed.",
        metavar="24-20160314.n.0",
    )
    parser_compose.add_argument("-y", "--testtype", help="The test type to generate a page for")
    parser_compose.add_argument("-u", "--username", help=username_help)
    parser_compose.add_argument("-f", "--force", help=force_help, action="store_true")
    parser_compose.add_argument("-t", "--test", help=test_help, action="store_true")
    parser_compose.add_argument(
        "-n",
        "--no-current",
        help="Do not update Test Results:Current redirect pages",
        action="store_true",
    )
    parser_compose.add_argument(
        "-w",
        "--download-only",
        help="Only write/update the Download page, do not touch any other page",
        action="store_true",
    )
    parser_compose.set_defaults(func=create_compose, login=True, guessrel=True)

    parser_userstats = subparsers.add_parser(
        "user-stats",
        description="Output statistics regarding how many tests "
        "were run and bugs reported by each individual tester for a set of "
        "test events.",
    )
    parser_userstats.add_argument(
        "-r",
        "--release",
        help=release_help,
        type=int,
        metavar="12-99",
        choices=list(range(12, 100)),
    )
    parser_userstats.add_argument(
        "-m", "--milestone", help=milestone_help_specific, choices=["Alpha", "Beta", "Final", "RC"]
    )
    parser_userstats.add_argument(
        "--modular",
        help="Operate on modular compose (deprecated, effect is same as --dist=Fedora-Modular)",
        action="store_true",
    )
    parser_userstats.add_argument(
        "--dist",
        help="Dist (compose shortname, e.g. Fedora or Fedora-IoT) to operate on",
        default="Fedora",
    )
    parser_userstats.add_argument("-t", "--test", help=test_help, action="store_true")
    parser_userstats.add_argument(
        "-f", "--filter", metavar="KEYWORD", help=filter_help, action="append"
    )
    parser_userstats.add_argument(
        "-s",
        "--since",
        help="Only include pages created on or after this date",
        metavar="YYYYMMDD",
    )
    parser_userstats.add_argument(
        "-u",
        "--until",
        help="Only include pages created before or on this date",
        metavar="YYYYMMDD",
    )
    parser_userstats.add_argument(
        "-b",
        "--bot",
        action="store_true",
        help="Include results from automated tests ('bot results')",
    )
    parser_userstats.set_defaults(func=user_stats, login=False, guessrel=True)

    parser_tcstats = subparsers.add_parser(
        "testcase-stats",
        description="Output a set of HTML pages containing "
        "tables providing an overview of test coverage across a series of "
        "test events.",
    )
    parser_tcstats.add_argument(
        "-r",
        "--release",
        help=release_help,
        type=int,
        metavar="12-99",
        choices=list(range(12, 100)),
    )
    parser_tcstats.add_argument(
        "-m", "--milestone", help=milestone_help_specific, choices=["Alpha", "Beta", "Final", "RC"]
    )
    parser_tcstats.add_argument(
        "--modular",
        help="Operate on modular compose (deprecated, effect is same as --dist=Fedora-Modular)",
        action="store_true",
    )
    parser_tcstats.add_argument(
        "--dist",
        help="Dist (compose shortname, e.g. Fedora or Fedora-IoT) to operate on",
        default="Fedora",
    )
    parser_tcstats.add_argument(
        "-o",
        "--out",
        help="The directory to write the pages to. WILL BE OVERWRITTEN if it already exists",
        metavar="/some/path",
    )
    parser_tcstats.add_argument("-t", "--test", help=test_help, action="store_true")
    parser_tcstats.add_argument(
        "-f", "--filter", metavar="KEYWORD", help=filter_help, action="append"
    )
    parser_tcstats.add_argument(
        "-n",
        "--nopostproc",
        help="Do not post-process the results to try "
        "and clean moved test cases and section order",
        action="store_true",
    )
    parser_tcstats.set_defaults(func=testcase_stats, login=False, guessrel=True)

    parser_result = subparsers.add_parser(
        "report-results",
        description="Submit validation test results "
        "for an event. This command requires login; see main help "
        "information about authentication.",
    )
    parser_result.add_argument("-u", "--username", help=username_help)
    parser_result.add_argument(
        "-r",
        "--release",
        help=release_help,
        type=int,
        metavar="12-99",
        choices=list(range(12, 100)),
    )
    parser_result.add_argument(
        "-m",
        "--milestone",
        help=milestone_help,
        choices=["Beta", "Final", "RC", "Branched", "Rawhide"],
    )
    parser_result.add_argument("-c", "--compose", help=compose_help, metavar="1.1 or 20160314.n.0")
    parser_result.add_argument(
        "--modular",
        help="Operate on modular compose (deprecated, effect is same as --dist=Fedora-Modular)",
        action="store_true",
    )
    parser_result.add_argument(
        "--dist",
        help="Dist (compose shortname, e.g. Fedora or Fedora-IoT) to operate on",
        default="Fedora",
    )
    parser_result.add_argument(
        "-y", "--testtype", help="The test type to report a result for (Base, Cloud...)"
    )
    parser_result.add_argument("-s", "--section", help="The page section to report the result in")
    parser_result.add_argument("-t", "--test", help=test_help, action="store_true")
    parser_result.set_defaults(func=report_results, login=True, guessrel=False)

    parser_sizecheck = subparsers.add_parser(
        "size-check", description="Check image sizes for a compose and report the results."
    )
    parser_sizecheck.add_argument("-u", "--username", help=username_help)
    parser_sizecheck.add_argument(
        "-r",
        "--release",
        help="The Fedora release to check",
        type=int,
        choices=list(range(1, 100)),
        metavar="1-99",
    )
    parser_sizecheck.add_argument(
        "-m",
        "--milestone",
        help="A milestone to check (e.g. Beta)",
        choices=["Beta", "Final", "RC", "Branched", "Rawhide"],
    )
    parser_sizecheck.add_argument(
        "-c", "--compose", help=compose_help, metavar="1.1 or 20160314.n.0"
    )
    parser_sizecheck.add_argument(
        "-i", "--cid", help="Compose ID of a Pungi 4 compose to check", metavar="24-20160314.n.0"
    )
    parser_sizecheck.add_argument(
        "--modular",
        help="Operate on modular compose (deprecated, effect is same as --dist=Fedora-Modular)",
        action="store_true",
    )
    parser_sizecheck.add_argument(
        "--dist",
        help="Dist (compose shortname, e.g. Fedora or Fedora-IoT) to operate on",
        default="Fedora",
    )
    parser_sizecheck.add_argument(
        "-b", "--bugzilla", help="Submit results to Bugzilla", action="store_true"
    )
    parser_sizecheck.add_argument("-t", "--test", help=test_help, action="store_true")
    parser_sizecheck.set_defaults(func=size_check, login=True, guessrel=False)

    args = parser.parse_args()
    # handle deprecated 'modular' arg
    if args.modular and args.dist == "Fedora":
        print(
            "WARNING: the --modular argument is deprecated and may be removed "
            "in a future release. Use --dist=Fedora-Modular instead."
        )
        args.dist = "Fedora-Modular"

    site = setup_site(args)
    # guessrel is kinda ugly, but some sub-functions don't want release
    # guessing
    if args.guessrel and not args.release and not getattr(args, "cid", ""):
        args.release = guess_release(site)
    if hasattr(args, "username") and args.username:
        print(
            "WARNING: the '--username' argument no longer has any effect "
            "and may be removed in a future release. See the README for "
            "more information on the newer auth system from 2018 onwards."
        )

    return (args, site)


## SUB-COMMAND METHODS ##


def create_compose(args, site):
    """Create the result pages for a given event (and, optionally,
    test type). Uses parameters passed in from the command line.
    """
    if not args.cid and not (args.release and args.milestone and args.compose):
        sys.exit("You must specify --cid or --milestone and --compose!")
    if args.cid and (args.release or args.milestone or args.compose):
        sys.exit(
            "You specified both --cid and at least one of --release, "
            "--milestone and --compose. This is ambiguous and not "
            "allowed. Please specify EITHER --cid OR --release, "
            "--milestone and --compose."
        )
    try:
        event = site.get_validation_event(
            args.release, args.milestone, args.compose, args.cid, dist=args.dist
        )
    except ValueError as err:
        sys.exit(err)

    if args.download_only:
        print("Only handling download page!")
        try:
            # handling force is...annoying. createonly must be 'None'
            # to force writing (not False)
            createonly = True
            if args.force:
                createonly = None
            event.download_page.write(createonly=createonly)
            return
        except mwe.APIError as err:
            sys.exit(err)

    if args.testtype:
        testtypes = [args.testtype]
    else:
        testtypes = None
    current = not args.no_current
    try:
        event.create(testtypes=testtypes, force=args.force, current=current)
    except mwe.APIError:
        print("Unhandled error writing pages!")
        raise
    except ValueError as err:
        sys.exit(err)


def user_stats(args, site):
    """Print HTML-formatted statistics on the number of results
    reported by each tester for a given set of result pages.
    """
    release = rl.Release(args.release, site, dist=args.dist)
    pages = release.milestone_pages(args.milestone)
    if args.filter:
        pages = [p for p in pages for f in args.filter if f in p.name]
    if args.since:
        pages = [p for p in pages if int(p.creation_date) >= int(args.since)]
    if args.until:
        pages = [p for p in pages if int(p.creation_date) <= int(args.until)]
    text = []
    # dump the results tables from each page into a single big string
    for page in pages:
        sys.stderr.write(f"Processing page: {page.name}\n")
        text.append(page.results_wikitext)
    text = "\n".join(text)

    # find results with the statuses we care about that are not transferred
    # from previous runs
    results = rs.find_results(
        text, statuses=["pass", "fail", "warn"], transferred=False, bot=args.bot
    )

    # make two defaultdicts for number of reports and cited bugs per unique
    # user. defaultdict(int) sets the value when adding a key to int(0).
    reports = defaultdict(int)
    bugs = defaultdict(set)
    for result in results:
        if result.user:
            reports[result.user.lower()] += 1
        # we don't want to include the bug if it's not just a digit
        # (e.g. comment in the wrong place, or smth like 'bgo#714')
        rbugs = [bug for bug in result.bugs if bug.isdigit()]
        if rbugs:
            bugs[result.user].update(rbugs)

    uss.print_stats(args.release, reports, bugs)


def testcase_stats(args, site):
    """Generate HTML-formatted statistics on the test coverage for
    a given release - produces summary pages for each test type
    containing tables with a row for each test indicating the level
    of coverage at each compose, and detail pages for each test
    with detailed statistics on the test results and bugs for that
    test for each compose.
    """
    outdir = tempfile.mkdtemp(prefix="tc_report_")
    release = rl.Release(args.release, site, dist=args.dist)
    relpages = release.milestone_pages(args.milestone)
    html_output = {}
    json_output = {}
    if args.filter:
        relpages = [p for p in relpages for f in args.filter if f in p.name]

    allpages = defaultdict(list)
    # for sorting purposes, we start out with the record of composes
    # as a dict, key is the compose version, value is its date
    allcomposes = {}
    for page in relpages:
        allpages[page.testtype].append(page)
        if page.shortver not in allcomposes:
            allcomposes[page.shortver] = page.creation_date

    # turn the composes dict into a list and sort it by creation date,
    # with wikitcms sort as a tiebreaker (viz 23 Final RC7 vs. RC9)
    allcomposes = list(allcomposes.items())
    allcomposes.sort(key=lambda x: (x[1], hl.fedora_release_sort(x[0])))
    allcomposes = [comp[0] for comp in allcomposes]

    # for each test type, sort the pages into the correct Fedora release
    # order, then extract the result rows from each page.
    #
    # 'tests' is an ordered dict of Test() objects, keyed with a tuple
    # identifying 'unique tests': the test 'name', its testcase page name,
    # and the name of the page section in which it appears. It is ordered
    # so that tests appear in the order they are encountered - when the
    # HTML output is rendered, the ordering of tests will be more or less
    # the same as on the actual result wiki pages. tcs.post_process()
    # cleans this up some more once all result pages have been parsed. See
    # its docs for details.
    #
    # A Test() object is generated for each result row from each result
    # page for the test type, and added to the tests dict based on those
    # three attributes (as the second field of a tuple, the first field
    # identifying the compose from which it comes). Result rows for which
    # all three attributes match are grouped together - they're considered
    # to be results for the same test in different composes.
    jsondata = {"allcomposes": allcomposes, "stats": {}}
    collected_pages = {}
    collected_json = {}
    for index, (testtype, pages) in enumerate(allpages.items()):
        print(f"Processing type [{index + 1}/{len(allpages)}]: {testtype}")
        tests = OrderedDict()
        # re-use the allcomposes sort to sort the pages.
        pages.sort(key=lambda x: allcomposes.index(x.shortver))
        for index, page in enumerate(pages):
            print(f"Processing page [{index + 1}/{len(pages)}]: {page.name}")
            resultrows = page.get_resultrows(statuses=["pass", "warn", "fail"], transferred=False)
            for row in resultrows:
                tests.setdefault((row.name, row.testcase, row.section), tcs.Test())
                tests[(row.name, row.testcase, row.section)].update(
                    page.shortver, row, page.base_name
                )

        if args.nopostproc:
            print("nopostproc passed! Skipping post-process.")
        else:
            print("Post-processing results...")
            tests = tcs.post_process(tests, allcomposes)
        data = tcs.prepare_results_data(testtype, pages, allcomposes, tests, "html")
        jsondata = tcs.prepare_results_data(testtype, page, allcomposes, tests, "json")
        # Construct the final html output
        html_output["timestamp"] = data["timestamp"]
        collected_pages[testtype] = data["sections"]
        # Construct the final json output
        json_output["timestamp"] = jsondata["timestamp"]
        collected_json[testtype] = jsondata["sections"]

    html_output["pages"] = collected_pages
    json_output["pages"] = collected_json

    # Print output to a temporary directory
    jsontarget = os.path.join(outdir, "data.json")
    with open(jsontarget, "w", encoding="utf-8") as outfile:
        json.dump(json_output, outfile, sort_keys=True, indent=4)

    htmltarget = os.path.join(outdir, "index.html")
    tcs.print_results_html(html_output, htmltarget)

    if args.out:
        try:
            shutil.rmtree(args.out)
        except FileNotFoundError:
            pass
        except PermissionError:
            logger.warning("Could not clean output dir %s!", args.out)
        try:
            shutil.move(outdir, args.out)
            print(f"All output moved to: {args.out}")
            print(f"Index page at: {os.path.join(args.out, 'index.html')}")
            print(f"JSON data at: {os.path.join(args.out, 'data.json')}")
        except (PermissionError, FileExistsError):
            print(f"Could not write to: {args.out}! Output is at: {outdir}")
    else:
        print(f"HTML Output: {htmltarget}")
        print(f"JSON Output: {jsontarget}")


def report_results(args, site):
    """Report some validation testing results."""
    rrs.report_results(
        site, args.release, args.milestone, args.compose, args.testtype, dist=args.dist
    )


def size_check(args, site):
    """Check sizes of images for event's compose and report results."""
    try:
        event = site.get_validation_event(
            args.release, args.milestone, args.compose, args.cid, dist=args.dist
        )
        ffrel = event.ff_release
        print(f"Checking sizes for event: {event.version}, compose ID: {ffrel.cid}")
    except ValueError:
        sys.exit("Could not find validation test event!")
    results = rsc.size_check(ffrel)
    if not results:
        sys.exit("No images found for event!")

    # submit results to wiki
    reslist = []
    for env, result in list(results.items()):
        if result == "pass":
            restup = wk.ResTuple(
                "Installation",
                args.release,
                args.milestone,
                args.compose,
                "ISO_Size",
                "",
                "",
                env,
                result,
                args.username,
                "",
                "",
                True,
                args.cid,
                args.dist,
            )
        else:
            # Fail - construct a comment with details
            comms = []
            for img, imgsize, maxsize in result:
                desc = " ".join((img["subvariant"], img["type"], img["arch"]))
                comms.append(f"{desc}, size {imgsize}, max {maxsize}")
            comment = f"<ref>{'</ref><ref>'.join(comms)}</ref>"
            # Mark this as a bot result as it's an automated test.
            restup = wk.ResTuple(
                "Installation",
                args.release,
                args.milestone,
                args.compose,
                "ISO_Size",
                "",
                "",
                env,
                "fail",
                args.username,
                "",
                comment,
                True,
                args.cid,
                args.dist,
            )
        reslist.append(restup)
    (insuffs, dupes) = site.report_validation_results(reslist)
    if dupes:
        for dupe in dupes:
            print(
                f"This user already reported a result for {dupe.env}! Will not "
                "file duplicate report."
            )
    #    if insuffs and len(insuffs) == len(results):
    #        sys.exit("Could not find result page or test case!")
    if insuffs:
        for insuff in insuffs:
            print(f"Could not find environment {insuff.env}!")

    if not args.bugzilla:
        # we're done!
        sys.exit(0)

    # login to Bugzilla
    if args.test:
        bzapi = bugzilla.Bugzilla("bugzilla.stage.redhat.com")
    else:
        bzapi = bugzilla.Bugzilla("bugzilla.redhat.com")
    # this solves a problem with Bug instances returned by build_createbug
    # not knowing what their aliases are...
    bzapi.bug_autorefresh = True
    if not bzapi.logged_in:
        bzapi.interactive_login()

    # find blocker trackers
    finalid = 0
    query = {
        "query_format": "advanced",
        "o1": "anywordssubstr",
        "f1": "alias",
        "v1": "F{0}FinalBlocker".format(event.release),
        "include_fields": ["id", "alias"],
    }
    ret = bzapi.query(query)
    for tracker in ret:
        if "FinalBlocker" in " ".join(tracker.alias):
            finalid = tracker.id
    if not finalid:
        # this is unexpected, warn about it
        logger.warning("No Final blocker tracker found for %s", event.release)

    # submit results to Bugzilla
    for env, result in list(results.items()):
        if result == "pass":
            continue
        for img, imgsize, maxsize in result:
            desc = " ".join((img["subvariant"], img["type"], img["arch"]))
            # maximum length of an alias is 40 characters
            aliasdesc = desc.replace(" ", "")[:28]
            alias = f"F{event.release}{aliasdesc}Oversize"
            summary = f"Fedora {event.release}: {desc} image exceeds maximum size"
            comment = "{0} image {1} from compose {2} is {3} bytes, exceeding the maximum size {4}."
            comment = comment.format(desc, img["url"], ffrel.cid, imgsize, maxsize)
            query = {
                "query_format": "advanced",
                "o1": "anywordssubstr",
                "f1": "alias",
                "v1": alias,
                "include_fields": ["id", "alias"],
            }
            ret = bzapi.query(query)
            if ret:
                # bug already exists, add a comment and set ASSIGNED
                bug = ret[0]
                print(f"Updating bug {bug.id} for over-size image {desc}")
                update = bzapi.build_update(
                    comment=comment,
                    status="ASSIGNED",
                )
                try:
                    # this can fail saying we don't have the right to change
                    # the status
                    bzapi.update_bugs([bug.id], update)
                except Fault:
                    # try with NEW?
                    update = bzapi.build_update(
                        comment=comment,
                        status="NEW",
                    )
                    try:
                        # this can still fail
                        bzapi.update_bugs([bug.id], update)
                    except Fault:
                        # fine just the comment
                        update = bzapi.build_update(
                            comment=comment,
                        )
                        bzapi.update_bugs([bug.id], update)
            else:
                # create new bug
                version = event.release
                if event.milestone.lower() == "rawhide":
                    version = "rawhide"
                description = comment + (
                    " Canonical maximum sizes can be found at "
                    "https://docs.fedoraproject.org/en-US/releases/f{0}/spins/ and "
                    "https://docs.fedoraproject.org/en-US/releases/f{0}/blocking/ "
                    ". This check is run by the 'relval' tool, which has its "
                    "own list of maximum sizes derived from those pages. If the "
                    "maximum size used for this comparison is wrong, please add "
                    "a comment and file a bug against relval at "
                    "https://pagure.io/fedora-qa/relval/issues and it will be "
                    "corrected. If you believe the canonical maximum size for "
                    "an image should be changed, please follow the appropriate "
                    "process before filing a relval bug."
                ).format(event.release)
                # canonical definition here is
                # https://docs.fedoraproject.org/en-US/releases/f40/blocking/
                # (swap '40' for current release). This is just a copy of
                # that info, because it's not programmatically consumable
                blocking = [
                    "Everything boot x86_64",
                    "Server dvd aarch64",
                    "Server boot aarch64",
                    "Server dvd x86_64",
                    "Server boot x86_64",
                    "Minimal raw-xz armhfp",
                    "Workstation raw-xz aarch64",
                    "KDE live x86_64",
                    "Workstation live x86_64",
                    "Container_Toolbox docker aarch64",
                    "Container_Toolbox docker x86_64",
                ]
                # this is the FedoraOversizeTracker, which exists for tracking
                # bugs filed by this tool
                blocks = [2041649]
                if desc in blocking and finalid:
                    blocks.append(finalid)
                print(f"Creating new bug alias {alias} for over-size image {desc}")
                create = bzapi.build_createbug(
                    product="Fedora",
                    component="distribution",
                    summary=summary,
                    version=version,
                    description=description,
                    alias=alias,
                    blocks=blocks,
                )
                logger.debug("Bug properties: %s", create)
                bzapi.createbug(create)

    sys.exit(0)


## MAIN LOOP BITS ##


def main():
    """Main loop"""
    try:
        args, site = parse_args()
        loglevel = getattr(logging, args.loglevel.upper(), logging.INFO)
        logging.basicConfig(level=loglevel)
        args.func(args, site)
    except KeyboardInterrupt:
        print("Interrupted, exiting...")
        sys.exit(1)


if __name__ == "__main__":
    main()
