# Copyright Red Hat
#
# This file is part of relval.
#
# wikitcms is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors:
# Adam Williamson <awilliam@redhat.com>

# this code is kinda irredeemable from these perspectives
# pylint: disable=too-many-locals,too-many-statements,too-many-branches,too-many-arguments

"""This file contains functions that help implement the report-results
sub-command of relval, which is a simple utility to report release
validation results.
"""

import re
import sys

import wikitcms.result
from productmd.composeinfo import get_date_type_respin

# these are some values we cache across multiple runs of the main
# function.
RESPAGES = None
PAGE = None
SECTIONS = None


def comment_string(string, maxlen=250):
    """Take 'string' and wrap it in <ref> </ref> if it's not the empty
    string. Raise a ValueError if it's longer than maxlen.
    """
    string = string.strip()
    if maxlen and len(string) > maxlen:
        err = f"Comment is too long: {len(string)} characters, max is {maxlen}."
        raise ValueError(err)
    # Don't produce an empty <ref></ref> if there's no comment
    if not string:
        return string
    return f"<ref>{string}</ref>"


def check_nightly(string):
    """Check if a string looks like a valid nightly identifier."""
    try:
        (_, typ, _) = get_date_type_respin(string)
        return typ == "nightly"
    except ValueError:
        return False


# mini functions for prompting the user for choices


def get_response(prompt, helptext, accepted=None, accfunc=None, backstr=None):
    """Get a response to a question. Prompt is the initial text,
    helptext is help text shown on invalid response, accepted is an
    iterable of valid responses (case-insensitive comparison and
    return). accfunc is a function to call on the response to
    validate it. backstr is a string to accept as back, or None if
    going back is not allowed. Either accepted or accfunc must be given.
    """

    if not accepted and not accfunc:
        raise ValueError("get_response: Must specify accepted or accfunc!")

    def check(resp, accepted, accfunc, backstr):
        """'accepted' makes things a mite tricky."""
        checkpassed = False

        if accfunc:
            checkpassed = checkpassed or accfunc(resp.lower())

        if accepted:
            checkpassed = checkpassed or resp.lower() in accepted

        if backstr:
            checkpassed = checkpassed or resp.lower() == backstr

        return checkpassed

    resp = input(prompt)
    while not check(resp, accepted, accfunc, backstr):
        print(f"Invalid response. {helptext}")
        resp = input(prompt)
    return resp.lower()


def get_confirm(backstr=None):
    """Just ask "yes" or "no" and return 'y' or 'n', or backstr
    if allowed."""
    prompt = "Confirm y/n? "
    helptext = "Answer y or n"

    if backstr:
        prompt = f"Confirm y/n (or {backstr} to go back)? "
        helptext += f" (or {backstr} to go back)"

    return get_response(prompt, helptext=helptext, accepted=("y", "n"), backstr=backstr)


def get_response_list(prompt, items, backstr=None):
    """Given a question and an iterable of strings, number the strings
    and ask the user which they want. Returns the int for the user's
    selection, minus 1 (because you probably want to number from 1 for
    humans but use the result as a list index), or backstr if allowed.
    """
    lines = []
    for num, item in enumerate(items, 1):
        lines.append(f"{num})  {item}")

    helptext = f"Pick a number between 1 and {len(items)}"

    if backstr:
        lines.append(f"{backstr}) Go back")
        helptext += f" (or {backstr} to go back)"

    lines.append(prompt)
    prompt = "\n".join(lines)

    accepted = [str(num) for num in range(1, len(items) + 1)]

    responsestr = get_response(prompt, helptext, accepted=accepted, backstr=backstr)

    if backstr and responsestr == backstr:
        return responsestr

    return int(responsestr) - 1


def get_result(username):
    """Get a wikitcms Result instance by asking the user for all the
    required information.
    """
    prompt = "Would you like to report a (p)ass, (f)ail or (w)arn? "
    helptext = "Please choose (p)ass, (f)ail or (w)arn."
    status = get_response(prompt, helptext, accepted=("p", "f", "w"))
    for let, full in (("p", "pass"), ("f", "fail"), ("w", "warn")):
        if status == let:
            status = full

    bugs = input("Enter the IDs of any bugs associated with this report, separated by commas. ")
    if bugs:
        # filter all whitespace
        bugs = re.sub(r"\s", "", bugs)
        # split on commas
        bugs = bugs.split(",")
        # drop any empty strings
        bugs = [bug for bug in bugs if bug]
    if not bugs:
        # turn '' or [] into None
        bugs = None
    comment = ""
    comm = input(
        "Enter a comment if you like. If not, just hit Enter.\n"
        "Maximum length is 250 characters.\nPlease don't leave a "
        "comment unless it's really necessary.\nComment: "
    )
    while comm:
        try:
            comment = comment_string(comm, 250)
            comm = ""
        except ValueError:
            comm = input(
                f"Comment was too long: {len(comm.strip())} characters. Please make "
                "it 250 characters or fewer.\n"
                "Comment: "
            )
    return wikitcms.result.Result(status, username, bugs, comment)


def report_results(
    wiki,
    release=None,
    milestone="",
    compose="",
    testtype=None,
    section=None,
    auto=True,
    checkver=True,
    dist=None,
):
    """Main function, find a test instance to report a result for and
    create a result, then report it.
    """
    username = wiki.username.lower()
    # these are global so they can be cached between runs
    global RESPAGES  # pylint:disable=global-statement
    global PAGE  # pylint:disable=global-statement
    global SECTIONS  # pylint:disable=global-statement
    version_set = False
    version_type = ""

    # this means 'try and detect the current compose, unless we already
    # have one fully specified or we've been told not to'
    if not (release and compose) and auto is True:
        if release or compose:
            print(
                "When specifying the compose version by parameters, you "
                "must pass at least a release and a compose. Your "
                "version-related parameters will not be used."
            )
        print("Detecting current compose version...")
        if not dist:
            # let's not ask, let's just guess the main stream
            dist = "Fedora"
        curr = wiki.get_current_compose(dist=dist)
        if curr:
            release = int(curr["release"])
            milestone = curr["milestone"]
            # this is a handy way to get whichever is set, and if for some
            # crazy reason both are set it'll prefer 'T/RCx' to 'YYYYMMDD',
            # which we probably want.
            compose = max(curr["compose"], curr["date"])

    if release and compose:
        version_set = True

    if dist is None:
        prompt = "What compose type are you reporting results for? "
        dists = ("Fedora", "Fedora-IoT")
        dist = dists[get_response_list(prompt, dists)]

    validrels = [str(num) for num in range(23, 100)]
    if str(release) not in validrels:
        prompt = "Which Fedora release would you like to report a result for? "
        helptext = "Please input a valid Fedora release number."
        release = int(get_response(prompt, helptext, accepted=validrels))

    while not version_set:
        prompt = (
            "Would you like to report a result for a milestone "
            "(c)ompose or a (n)ightly compose? "
        )
        helptext = "Please choose (c)ompose or (n)ightly."
        version_type = get_response(prompt, helptext, accepted=("c", "n"))

        if version_type == "c":
            prompt = "(B)eta\n(R)C\nWhich milestone would you like to report a milestone for? "
            helptext = "Please choose (B)eta, or (R)C."
            milestone = get_response(prompt, helptext, accepted=("b", "r"))
            for let, full in (("b", "Beta"), ("r", "RC")):
                if milestone == let:
                    milestone = full
            if not compose:
                prompt = "Which compose would you like to report a result for? "
                helptext = "Examples: 1.0, 1.1, 2.1"
                compose_patt = re.compile(r"\d+\.\d+")
                compose = get_response(prompt, helptext, accfunc=compose_patt.match).upper()
            version_set = True
        else:
            if not compose:
                prompt = "What nightly would you like to report a result for? "
                helptext = "Nightly must be YYYYMMDD.n.N (e.g. 20160314.n.0)"
                compose = get_response(prompt, helptext, accfunc=check_nightly)
            version_set = True

    if milestone:
        dispver = f"{dist} {release} {milestone} {compose}"
    else:
        dispver = f"{dist} {release} {compose}"
    if checkver:
        print(f"Will report results for compose: {dispver}")
        confirm = get_confirm()
        if confirm != "y":
            report_results(wiki, auto=False)

    if not RESPAGES:
        try:
            event = wiki.get_validation_event(
                release=release, milestone=milestone, compose=compose, dist=dist
            )
        except ValueError as err:
            print(err)
            report_results(wiki)

        RESPAGES = event.result_pages
        if not RESPAGES:
            print("Error! You probably chose a compose that doesn't exist.")
            report_results(wiki)

    if PAGE is None:
        if testtype:
            try:
                PAGE = [pag for pag in RESPAGES if pag.testtype == testtype][0]
            except IndexError:
                print("Error! Specified test type does not exist.")
        if not PAGE:
            prompt = "Which test type would you like to report a result for? "
            items = [page.testtype for page in RESPAGES]

            pageresponse = get_response_list(prompt, items, backstr="back")

            if pageresponse == "back":
                # Go back to confirming selected compose
                report_results(wiki)

            PAGE = RESPAGES[pageresponse]

    tests = PAGE.get_resultrows(statuses=["pass", "warn", "fail", "inprogress"])

    if not section:
        try:
            # This serves two purposes: if we somehow wound up with a dead
            # page, we can try again...
            if len(SECTIONS) == 0:
                print("Error! No results found for this page.")
                report_results(wiki)
        except TypeError:
            # ...and if we hit an exception, it means we need to get SECTIONS.
            SECTIONS = PAGE.results_sections
            # Filter out sections with no tests in them. wikitcms should
            # probably do this but it's actually not trivial for $REASONS
            testsecs = {t.secid for t in tests}
            SECTIONS = [s for s in SECTIONS if s["index"] in testsecs]
        if len(SECTIONS) == 1:
            section = SECTIONS[0]
        else:
            prompt = "Which section would you like to report a result in? "
            tag_patt = re.compile("<.*?>")
            items = [tag_patt.sub("", section["line"]) for section in SECTIONS]

            sectionresponse = get_response_list(prompt, items, backstr="back")

            if sectionresponse == "back":
                # Go back to selecting a page
                PAGE = None
                SECTIONS = None
                report_results(wiki, release, milestone, compose, checkver=False, dist=dist)

            section = SECTIONS[sectionresponse]

    # Find the tests for the section the user chose.
    tests = [t for t in tests if t.secid == section["index"]]
    if len(tests) == 1:
        test = tests[0]
    else:
        prompt = "Which test would you like to report a result for? "
        items = []
        for candtest in tests:
            if candtest.name != candtest.testcase:
                items.append(f"{candtest.testcase} {candtest.name}")
            else:
                items.append(candtest.testcase)

        testresponse = get_response_list(prompt, items, backstr="back")

        if testresponse == "back":
            # Go back to selecting a section
            report_results(wiki, release, milestone, compose, checkver=False, dist=dist)

        test = tests[testresponse]

    envs = list(test.results.keys())
    if len(envs) == 1:
        env = envs[0]
    elif len(envs) > 1:
        prompt = "Which environment would you like to report a result for? "

        environmentresponse = get_response_list(prompt, envs, backstr="back")

        if environmentresponse == "back":
            # Go back to selecting a test
            report_results(
                wiki, release, milestone, compose, testtype, section, checkver=False, dist=dist
            )

        env = envs[environmentresponse]

    results = test.results[env]
    if results:
        print("Current results for this test:")
        for result in results:
            print(str(result))
        myres = [r for r in results if r.user == username]
        if myres:
            print(
                "You have already reported result(s) for this test! Do you "
                "still want to report this result?"
            )
            confirm = get_confirm()
            if confirm == "y":
                pass
            else:
                report_results(wiki, release, milestone, compose, dist=dist)

    res = get_result(username)
    print(
        f"Will report {res} for test {test.testcase} in section "
        f"{section['line']} of {PAGE.name}"
    )
    PAGE.add_result(res, test, env)
    print("Succeeded!")

    # we're done, what next?
    prompt = "Report another result for this (s)ection, or (p)age, or this (c)ompose, or (q)uit? "
    helptext = "Please choose (s)ection, (p)age, (c)ompose, or (q)uit."
    cont = get_response(prompt, helptext, ("s", "p", "c", "q"))
    if cont == "q":
        sys.exit()
    else:
        test = None
    if cont in ("p", "c"):
        section = None
    if cont == "c":
        testtype = None
        PAGE = None
        SECTIONS = None
    report_results(wiki, release, milestone, compose, testtype, section, checkver=False, dist=dist)
