# Copyright Red Hat
#
# This file is part of relval.
#
# wikitcms is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Adam Williamson <awilliam@redhat.com>
#
"""This file contains functions that help implement the 'size_check'
sub-command of relval, which checks the sizes of images and reports
the results.
"""

import fedfind.helpers
import fedfind.release

STDTYPES = ("netinst", "boot", "dvd", "live", "dvd-ostree", "raw-xz", "container")


def size_check(ffrel):
    """Do the thing! Check the things! CHECK ALL THE THINGS"""
    # get the maximum size data from the server we keep it on. Refs:
    # https://fedoraproject.org/wiki/QA:Testcase_Mediakit_ISO_Size
    # https://fedoraproject.org/wiki/Releases/34/Spins
    # https://fedoraproject.org/wiki/Releases/34/ReleaseBlocking
    sizeurl = "https://fedorapeople.org/groups/qa/metadata/relvalsizes.json"
    maxsizes = fedfind.helpers.download_json(sizeurl)
    results = {}
    # these are all the variants we have target sizes for
    for variant in ("Everything", "Server", "Workstation", "Cloud", "Spins", "Labs", "Container"):
        print(f"Checking {variant}...")
        fails = []
        imgs = [img for img in ffrel.all_images if img["variant"] == variant]
        imgs = [img for img in imgs if img["type"] in STDTYPES]
        if variant in ("Spins", "Labs"):
            imgs = [img for img in imgs if img["type"] in ("live", "raw-xz")]
        if not imgs:
            print("No images found!")
            continue

        for img in imgs:
            maxsize = 4700000000
            maxsize = maxsizes.get(img["type"], maxsize)
            maxsize = maxsizes.get(img["subvariant"].lower(), maxsize)
            maxsize = maxsizes.get("_".join((img["subvariant"].lower(), img["type"])), maxsize)
            maxsize = maxsizes.get(
                "_".join((img["subvariant"].lower(), img["type"], img["arch"])), maxsize
            )
            url = "/".join((ffrel.location, img["path"]))
            imgsize = fedfind.helpers.get_size(url)
            if imgsize > maxsize:
                fails.append((img, imgsize, maxsize))

        if fails:
            results[variant] = fails
            for gotimg, gotsize, gotmax in fails:
                filename = gotimg["path"].split("/")[-1]
                print(f"FAIL: {filename}, size {gotsize}, max {gotmax}")
        else:
            results[variant] = "pass"
            print("PASS!")
    return results
