# Copyright Red Hat
#
# This file is part of relval.
#
# wikitcms is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors:
# Adam Williamson <awilliam@redhat.com>
# Based on initial implementation by Kamil Paral <kparal@redhat.com>
# Original at https://git.fedorahosted.org/cgit/fedora-qa.git/tree/stats

"""This file contains functions that help implement the 'userstats'
sub-command of relval, which produces statistics about user
contributions to release validation testing in HTML format.
"""

from markupsafe import escape


def print_stats(release, reports, bugs):
    """Prints the results for the 'user_stats' function in an HTML format."""

    template = """\
<!DOCTYPE HTML>
<html>
<head><meta http-equiv="Content-Type" content="text/html; charset=UTF-8"/></head>
<body>
  <p>Test period: <b>Fedora {release} [FIXME] - Fedora {release} [FIXME]</b><br/> \
     Testers: <b>{all_testers}</b><br/> \
     Reports: <b>{all_reports}</b><br/> \
     Unique referenced bugs: <b>{all_bugs}</b> \
  </p>
  <table>
    <tr style="text-align:center">
        <th style="vertical-align:middle">Name</th>
        <th style="vertical-align:middle">Reports submitted</th>
        <th style="vertical-align:middle">Referenced bugs<a style="text-decoration:none" \
        href="#ref"><sup>1</sup></a></th>
    </tr>
{rows}
  </table>
  <p style="color:#808080;font-size:smaller"> \
  <sup id="ref">1</sup> This is a list of bug reports <i>referenced</i> in test \
  results. The bug itself may not be created by the same person. \
  </p>
</body>
</html>
"""

    # sort according to score
    ladder = list(reports.keys())
    ladder.sort(key=lambda name: reports[name], reverse=True)

    # format results
    rows = []
    for name in ladder:
        capname = name.capitalize()
        trow = "<tr>"
        # name
        trow += f'<td><a href="https://fedoraproject.org/wiki/User:{escape(capname)}">'
        trow += f"{escape(name)}</a></td>"
        # reports
        trow += f'<td align="center">{reports[name]}</td>'
        # bugs
        if name in bugs:
            trtemp = '<td><span style="font-size:smaller">{}</span> ({})</td>'
            urltemp = '<a href="https://bugzilla.redhat.com/show_bug.cgi?id={0}">{0}</a>'
            ourbugs = sorted(bugs[name])
            buglinks = [urltemp.format(escape(bug)) for bug in ourbugs]
            trow += trtemp.format(" ".join(buglinks), len(ourbugs))
        else:
            trow += "<td/>"
        trow += "</tr>"
        rows.append(trow)

    rows = "\n".join(rows)

    all_bugs = 0
    if ladder and bugs:
        all_bugs = len(set().union(*bugs.values()))
    output = template.format(
        release=release,
        all_testers=len(ladder),
        all_reports=sum(reports[name] for name in ladder),
        all_bugs=all_bugs,
        rows=rows,
    )

    # print
    print(output)
