# Copyright Red Hat
#
# This file is part of relval.
#
# relval is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors:
# Brandon Nielsen <nielsenb@jetfuse.net>
# Adam Williamson <awilliam@redhat.com>

# these are all kinda inappropriate for pytest patterns
# pylint:disable=too-few-public-methods,too-many-public-methods,too-many-lines,too-many-arguments
# pylint:disable=missing-function-docstring,missing-class-docstring
# breaking up the long strings doesn't really help readability


"""Tests for report_results.py."""

from unittest.mock import Mock, call, patch

import pytest

from relval.report_results import (
    check_nightly,
    comment_string,
    get_confirm,
    get_response,
    get_response_list,
    get_result,
    report_results,
)

import sys

# long string constants, to make pylint happy
ALREADY = (
    "You have already reported result(s) for this test! Do you still want to report this result?"
)
ANOTHER = "Report another result for this (s)ection, or (p)age, or this (c)ompose, or (q)uit? "
ATLEAST = (
    "When specifying the compose version by parameters, you must pass at least a release "
    "and a compose. Your version-related parameters will not be used."
)
ENTERCOMMENT = (
    "Enter a comment if you like. If not, just hit Enter.\n"
    "Maximum length is 250 characters.\nPlease don't leave a comment "
    "unless it's really necessary.\nComment: "
)
MSORNIGHTLY = "Would you like to report a result for a milestone (c)ompose or a (n)ightly compose? "


# A decorator so we can maintain a reference to a patched
# function and call it
def reference_wrapper(function):
    def wrapped(*args, **kwargs):
        return function(*args, **kwargs)

    return wrapped


# Crude checking a single recursive report_results call
def report_results_exit(*args, **kwargs):
    sys.exit()


class TestCommentString:
    def test_basic_functionality(self):
        assert comment_string("test") == "<ref>test</ref>"

        # Make sure the string is stripped
        assert comment_string("\ttest\n") == "<ref>test</ref>"
        assert comment_string(" test ") == "<ref>test</ref>"

    def test_empty_string(self):
        assert comment_string("") == ""

    def test_maxlen(self):
        # Repeat "a" 251 times, check exception
        with pytest.raises(ValueError):
            comment_string("a" * 251)

        with pytest.raises(ValueError):
            comment_string("test", maxlen=3)


class TestCheckNightly:
    @patch("relval.report_results.get_date_type_respin")
    def test_nightly(self, mock_get_respin):
        mock_get_respin.return_value = ("", "nightly", "")

        assert check_nightly("teststring")

        mock_get_respin.assert_called_with("teststring")

    @patch("relval.report_results.get_date_type_respin")
    def test_rc(self, mock_get_respin):
        mock_get_respin.return_value = ("", "RC", "")

        assert not check_nightly("teststring")

        mock_get_respin.assert_called_with("teststring")

    @patch("relval.report_results.get_date_type_respin")
    def test_exception(self, mock_get_respin):
        mock_get_respin.side_effect = ValueError()

        assert not check_nightly("teststring")

        mock_get_respin.assert_called_with("teststring")


class TestGetResponse:
    @patch("relval.report_results.input")
    def test_no_accepted_or_accfunc(self, _):
        with pytest.raises(ValueError):
            get_response("test prompt", "test help text")

    @patch("relval.report_results.input")
    def test_accfunc(self, mock_input):
        # Verify a response can be accepted by accfunc
        mock_input.return_value = "test input"

        mock_acc_func = Mock()

        mock_acc_func.return_value = True

        result = get_response("test prompt", "", accfunc=mock_acc_func)

        assert result == "test input"

        mock_input.assert_called_with("test prompt")
        mock_acc_func.assert_called_with("test input")

    @patch("relval.report_results.input")
    def test_accfunc_case_insensitivity(self, mock_input):
        # Verify responses are case insensitive
        mock_input.return_value = "tEsT InPuT"

        mock_acc_func = Mock()

        mock_acc_func.return_value = True

        result = get_response("test prompt", "", accfunc=mock_acc_func)

        assert result == "test input"

        mock_input.assert_called_with("test prompt")
        mock_acc_func.assert_called_with("test input")

    @patch("relval.report_results.input")
    @patch("relval.report_results.print")
    def test_accfunc_not_accepted(self, mock_print, mock_input):
        # Don't accept once, accept the second time, ensure the user is
        # prompted a second time with help test
        mock_input.return_value = "test input"

        mock_acc_func = Mock()

        mock_acc_func.side_effect = [False, True]

        result = get_response("test prompt", "test help text", accfunc=mock_acc_func)

        assert result == "test input"

        assert mock_input.mock_calls == [call("test prompt"), call("test prompt")]
        assert mock_acc_func.mock_calls == [call("test input"), call("test input")]
        mock_print.assert_called_with("Invalid response. test help text")

    @patch("relval.report_results.input")
    def test_accepted(self, mock_input):
        # Verify a response can be accepted by a list of accepted values
        mock_input.return_value = "test input"

        result = get_response("test prompt", "", accepted=["test input"])

        assert result == "test input"

        mock_input.assert_called_with("test prompt")

    @patch("relval.report_results.input")
    @patch("relval.report_results.print")
    def test_accepted_not_accepted(self, mock_print, mock_input):
        # Invalid value once, valid the second time, ensure the user is
        # prompted a second time with help test
        mock_input.side_effect = ["invalid test input", "test input"]

        result = get_response("test prompt", "test help text", accepted=["test input"])

        assert result == "test input"

        assert mock_input.mock_calls == [call("test prompt"), call("test prompt")]
        mock_print.assert_called_with("Invalid response. test help text")

    @patch("relval.report_results.input")
    def test_accepted_case_insensitivity(self, mock_input):
        # Verify responses are case insensitive
        mock_input.return_value = "tEsT InPuT"

        result = get_response("test prompt", "", accepted=["test input"])

        assert result == "test input"

        mock_input.assert_called_with("test prompt")

    @patch("relval.report_results.input")
    def test_accfunc_backstr(self, mock_input):
        mock_input.return_value = "back"

        mock_acc_func = Mock()

        mock_acc_func.return_value = True

        result = get_response("test prompt", "", accfunc=mock_acc_func, backstr="back")

        assert result == "back"

        mock_input.assert_called_with("test prompt")
        mock_acc_func.assert_called_with("back")

    @patch("relval.report_results.input")
    def test_accepted_backstr(self, mock_input):
        mock_input.return_value = "back"

        result = get_response("test prompt", "", accepted=["test input"], backstr="back")

        assert result == "back"

        mock_input.assert_called_with("test prompt")


class TestGetConfirm:
    @patch("relval.report_results.get_response")
    def test_get_confirm(self, mock_get_response):
        # Verify get_confirm just gets a yes or no response
        result = get_confirm()

        assert result == mock_get_response.return_value

        mock_get_response.assert_called_with(
            "Confirm y/n? ", helptext="Answer y or n", accepted=("y", "n"), backstr=None
        )

    @patch("relval.report_results.get_response")
    def test_get_confirm_backstr(self, mock_get_response):
        # Make sure the backstr is handled correctly
        mock_get_response.return_value = "back"

        assert get_confirm(backstr="back") == "back"

        mock_get_response.assert_called_with(
            "Confirm y/n (or back to go back)? ",
            helptext="Answer y or n (or back to go back)",
            accepted=("y", "n"),
            backstr="back",
        )


class TestGetResponseList:
    @patch("relval.report_results.get_response")
    def test_get_response_list(self, mock_get_response):
        # Call twice to double check the result in converted to 0 index, and
        # cast to an int
        mock_get_response.side_effect = ["2", "1"]

        assert get_response_list("test prompt", ["test item 1", "test item 2"]) == 1
        assert get_response_list("test prompt", ["test item 1", "test item 2"]) == 0

        assert mock_get_response.mock_calls == [
            call(
                "1)  test item 1\n2)  test item 2\ntest prompt",
                "Pick a number between 1 and 2",
                accepted=["1", "2"],
                backstr=None,
            ),
            call(
                "1)  test item 1\n2)  test item 2\ntest prompt",
                "Pick a number between 1 and 2",
                accepted=["1", "2"],
                backstr=None,
            ),
        ]

    @patch("relval.report_results.get_response")
    def test_get_response_list_backstr(self, mock_get_response):
        # Make sure the backstr is handled correctly
        mock_get_response.return_value = "back"

        assert (
            get_response_list("test prompt", ["test item 1", "test item 2"], backstr="back")
            == "back"
        )

        mock_get_response.assert_called_with(
            "1)  test item 1\n2)  test item 2\nback) Go back\ntest prompt",
            "Pick a number between 1 and 2 (or back to go back)",
            accepted=["1", "2"],
            backstr="back",
        )


class TestGetResult:
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_pass(self, mock_result, mock_input, mock_get_response):
        # Test a pass, no issue, no comment
        mock_get_response.return_value = "p"

        mock_input.side_effect = ["", ""]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_get_response.assert_called_with(
            "Would you like to report a (p)ass, (f)ail or (w)arn? ",
            "Please choose (p)ass, (f)ail or (w)arn.",
            accepted=("p", "f", "w"),
        )
        assert mock_input.mock_calls == [
            call("Enter the IDs of any bugs associated with this report, separated by commas. "),
            call(ENTERCOMMENT),
        ]
        mock_result.assert_called_with("pass", "testusername", None, "")

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_fail(self, mock_result, mock_input, mock_get_response):
        # Test a fail, no issue, no comment
        mock_get_response.return_value = "f"

        mock_input.side_effect = ["", ""]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_get_response.assert_called_with(
            "Would you like to report a (p)ass, (f)ail or (w)arn? ",
            "Please choose (p)ass, (f)ail or (w)arn.",
            accepted=("p", "f", "w"),
        )
        assert mock_input.mock_calls == [
            call("Enter the IDs of any bugs associated with this report, separated by commas. "),
            call(ENTERCOMMENT),
        ]
        mock_result.assert_called_with("fail", "testusername", None, "")

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_warn(self, mock_result, mock_input, mock_get_response):
        # Test a warn, no issue, no comment
        mock_get_response.return_value = "w"

        mock_input.side_effect = ["", ""]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_get_response.assert_called_with(
            "Would you like to report a (p)ass, (f)ail or (w)arn? ",
            "Please choose (p)ass, (f)ail or (w)arn.",
            accepted=("p", "f", "w"),
        )
        assert mock_input.mock_calls == [
            call("Enter the IDs of any bugs associated with this report, separated by commas. "),
            call(ENTERCOMMENT),
        ]
        mock_result.assert_called_with("warn", "testusername", None, "")

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_bugs(self, mock_result, mock_input, mock_get_response):
        # Test "bugs" response
        mock_get_response.return_value = "f"

        mock_input.side_effect = ["1,2", ""]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_result.assert_called_with("fail", "testusername", ["1", "2"], "")

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_comment(self, mock_result, mock_input, mock_get_response):
        # Test comment, verify it's commented correctly
        mock_get_response.return_value = "p"

        mock_input.side_effect = ["", "test comment"]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_result.assert_called_with("pass", "testusername", None, "<ref>test comment</ref>")

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.input")
    @patch("relval.report_results.wikitcms.result.Result")
    def test_get_result_comment_length(self, mock_result, mock_input, mock_get_response):
        # Test comment, verify a reprompt if the comment is too long
        mock_get_response.return_value = "p"

        # Repeat "a" 251 times (over the limit), then try again
        mock_input.side_effect = ["", "a" * 251, "test comment"]

        result = get_result("testusername")

        assert result == mock_result.return_value

        mock_result.assert_called_with("pass", "testusername", None, "<ref>test comment</ref>")

        # Check for reprompt
        assert mock_input.mock_calls == [
            call("Enter the IDs of any bugs associated with this report, separated by commas. "),
            call(ENTERCOMMENT),
            call(
                "Comment was too long: 251 characters. Please make it 250 characters or fewer.\n"
                "Comment: "
            ),
        ]


class TestReportResult:
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_release_and_compose(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_one,
    ):
        # Test with release and compose provided
        mock_wiki = Mock()

        (mock_test_result, mock_page) = mock_page_one

        # "Fedora"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, release=99, compose="test compose")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response_list.mock_calls == [
            call("What compose type are you reporting results for? ", ("Fedora", "Fedora-IoT"))
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Will report results for compose: Fedora 99 test compose"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_release_only(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test with only release
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, release=97)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call(ATLEAST),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_compose_only(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test with only compose
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, compose="20221006")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call(ATLEAST),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.check_nightly")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_release_only_no_current_compose(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_check_nightly,
        mock_page_one,
    ):
        # Test with only release without a current compose
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = None

        (mock_test_result, mock_page) = mock_page_one

        # Nightly, 20221006.n.1, quit after get_result
        mock_get_response.side_effect = ["n", "20221006.n.1", "q"]

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        mock_check_nightly.return_value = True

        with pytest.raises(SystemExit):
            report_results(mock_wiki, release=97)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                "What nightly would you like to report a result for? ",
                "Nightly must be YYYYMMDD.n.N (e.g. 20160314.n.0)",
                accfunc=mock_check_nightly,
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call(ATLEAST),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 97 20221006.n.1"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.check_nightly")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_compose_only_no_current_compose(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_check_nightly,
        mock_page_one,
    ):
        # Test with only compose without a current compose
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = None

        (mock_test_result, mock_page) = mock_page_one

        # Release 97, nightly, quit after get_result
        mock_get_response.side_effect = ["97", "n", "q"]

        # Confirm release
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        mock_check_nightly.return_value = True

        with pytest.raises(SystemExit):
            report_results(mock_wiki, compose="20220708")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                "Which Fedora release would you like to report a result for? ",
                "Please input a valid Fedora release number.",
                accepted=[str(num) for num in range(23, 100)],
            ),
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call(ATLEAST),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 97 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.re")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_milestone(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_re,
        mock_page_one,
    ):
        # Test the milestone flow
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = None

        (mock_test_result, mock_page) = mock_page_one

        # Release 99, milestone (c)ompose, (b)eta, 2.1, quit after get_result
        mock_get_response.side_effect = ["99", "c", "b", "2.1", "q"]

        # Confirm release
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                "Which Fedora release would you like to report a result for? ",
                "Please input a valid Fedora release number.",
                accepted=[str(num) for num in range(23, 100)],
            ),
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                "(B)eta\n(R)C\nWhich milestone would you like to report a milestone for? ",
                "Please choose (B)eta, or (R)C.",
                accepted=("b", "r"),
            ),
            call(
                "Which compose would you like to report a result for? ",
                "Examples: 1.0, 1.1, 2.1",
                accfunc=mock_re.compile().match,
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 Beta 2.1"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_milestone_compose_set(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test the milestone flow with compose determined by get_current_compose
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 0,
            "milestone": "RC",
            "compose": "RC1",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Release 99, milestone (c)ompose, (b)eta, 2.1, quit after get_result
        mock_get_response.side_effect = ["99", "c", "b", "q"]

        # Confirm release
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response.mock_calls == [
            call(
                "Which Fedora release would you like to report a result for? ",
                "Please input a valid Fedora release number.",
                accepted=[str(num) for num in range(23, 100)],
            ),
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                "(B)eta\n(R)C\nWhich milestone would you like to report a milestone for? ",
                "Please choose (B)eta, or (R)C.",
                accepted=("b", "r"),
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 Beta RC1"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.check_nightly")
    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_confirm_retry(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_check_nightly,
        mock_page_one,
    ):
        # Test flow for not confirming compose, then confirming
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # "Fedora"
        mock_get_response_list.return_value = 0

        # 99, (n)ightly, 20160314.n.0, quit after get_result
        mock_get_response.side_effect = ["99", "n", "20160314.n.0", "q"]

        # Don't confirm, then confirm release 99
        mock_get_confirm.side_effect = ["n", "y"]

        # Result is "pass"
        mock_get_result.return_value = "pass"

        mock_check_nightly.return_value = True

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                "Which Fedora release would you like to report a result for? ",
                "Please input a valid Fedora release number.",
                accepted=[str(num) for num in range(23, 100)],
            ),
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                "What nightly would you like to report a result for? ",
                "Nightly must be YYYYMMDD.n.N (e.g. 20160314.n.0)",
                accfunc=mock_check_nightly,
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report results for compose: Fedora 99 20160314.n.0"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_auto_false(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_one,
    ):
        # Test with auto set to False
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # "Fedora"
        mock_get_response_list.return_value = 0

        # Release 99, (n)ightly, compose 20220711.n.0, quit after get_result
        mock_get_response.side_effect = ["99", "n", "20220711.n.0", "q"]

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, auto=False)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        assert mock_get_response_list.mock_calls == [
            call("What compose type are you reporting results for? ", ("Fedora", "Fedora-IoT"))
        ]

        assert mock_get_response.mock_calls == [
            call(
                "Which Fedora release would you like to report a result for? ",
                "Please input a valid Fedora release number.",
                accepted=[str(num) for num in range(23, 100)],
            ),
            call(
                MSORNIGHTLY,
                "Please choose (c)ompose or (n)ightly.",
                accepted=("c", "n"),
            ),
            call(
                "What nightly would you like to report a result for? ",
                "Nightly must be YYYYMMDD.n.N (e.g. 20160314.n.0)",
                accfunc=check_nightly,
            ),
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Will report results for compose: Fedora 99 20220711.n.0"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_dist(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, dist="testdist")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="testdist")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: testdist 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_checkver(
        self, mock_print, mock_get_result, mock_get_response, mock_page_one
    ):
        # Test with checkver=False
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, checkver=False)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_respages_testtype(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_event_res_pages
    ):
        # Test with no RESPAGES global but with a testtype provided
        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.return_value = mock_event

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, testtype="testtesttype")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")
        mock_wiki.get_validation_event.assert_called_with(
            release=99, milestone="test", compose="20220708", dist="Fedora"
        )

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_respages_bad_testtype(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_event_res_pages,
    ):
        # Test specifying a test type no page matches
        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.return_value = mock_event

        # "Fedora"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            # use a different test type to the one the page has
            report_results(mock_wiki, testtype="someothertesttype")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")
        mock_wiki.get_validation_event.assert_called_with(
            release=99, milestone="test", compose="20220708", dist="Fedora"
        )

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Error! Specified test type does not exist."),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_respages_get_validation_event_exception(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_event_res_pages,
    ):
        # Test with no RESPAGES global, and an exception with get_validation_event
        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        test_validation_error = ValueError()

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.side_effect = [test_validation_error, mock_event]

        # "Fedora"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, testtype="testtesttype")

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")
        mock_wiki.get_validation_event.assert_called_with(
            release=99, milestone="test", compose="20220708", dist="Fedora"
        )

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        # Confirm twice, first time resulted in an exception
        assert mock_get_confirm.mock_calls == [call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call(test_validation_error),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_respages(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_event_res_pages,
    ):
        # Test with no RESPAGES, and empty result pages
        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_wiki = Mock()

        mock_empty_event = Mock()
        mock_empty_event.result_pages = []

        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.side_effect = [mock_empty_event, mock_event]

        # "Fedora"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")
        mock_wiki.get_validation_event.assert_called_with(
            release=99, milestone="test", compose="20220708", dist="Fedora"
        )

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        # Confirm twice, first time resulted in an error
        assert mock_get_confirm.mock_calls == [call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Error! You probably chose a compose that doesn't exist."),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS")
    def test_report_results_no_sections(
        self,
        mock_sections,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_page_one,
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Hacky work around for mocking SECTIONS global checks
        # First we pretend it's empty to get the "No results found for this
        # page." error, then return 1 for the length when we hit that check
        # again, then once more to set section = SECTIONS[0]
        mock_sections.__len__.side_effect = [0, 1, 1]
        mock_sections.__getitem__.return_value = {"index": "test 1", "line": "test section"}

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        # Once with the error, once again with the successful run
        assert mock_get_confirm.mock_calls == [call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Error! No results found for this page."),
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", None)
    def test_report_results_sections_type_error(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Set SECTIONS to None, which should raise a TypeError when
        # checking the length, and re-fetch the sections from the page
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        mock_page.results_sections = [{"index": "test 1", "line": "test section"}]

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch(
        "relval.report_results.SECTIONS",
        [
            {"index": "test 1", "line": "test section 1"},
            {"index": "test 2", "line": "test section 2"},
        ],
    )
    def test_report_results_multiple_sections(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_one,
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        # Section "test 1"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which section would you like to report a result in? ",
                ["test section 1", "test section 2"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section 1 of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    def test_report_results_multiple_tests(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_two,
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test1_result, _, mock_page) = mock_page_two

        # Select "test case 1"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, section={"index": "test 1", "line": "test section"})

        mock_page.add_result.assert_called_with("pass", mock_test1_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test would you like to report a result for? ",
                ["test case test name 1", "test case test name 2"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    def test_report_results_multiple_tests_name_matches_testcase(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_two,
    ):
        # Test building selection list where test result testcase matches name
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test1_result, mock_test2_result, mock_page) = mock_page_two
        mock_test1_result.testcase = "test name 1"
        mock_test2_result.testcase = "test name 2"

        # Select "test case 1"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki, section={"index": "test 1", "line": "test section"})

        mock_page.add_result.assert_called_with("pass", mock_test1_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test would you like to report a result for? ",
                ["test name 1", "test name 2"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test name 1 in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_multiple_envs(
        self,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_get_response_list,
        mock_page_one,
    ):
        # Test with multiple envs
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one
        mock_test_result.results = {"testenv1": None, "testenv2": None}

        # Select "testenv1"
        mock_get_response_list.return_value = 0

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "testenv1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which environment would you like to report a result for? ",
                ["testenv1", "testenv2"],
                backstr="back",
            )
        ]

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_envs(self, mock_print, mock_get_confirm, mock_page_one):
        # Test with no envs
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one
        mock_test_result.results = {}

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        with pytest.raises(UnboundLocalError):
            report_results(mock_wiki)

        mock_page.add_result.not_called_with()

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_current_results(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test with already reported results
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        mock_current_result = Mock()
        mock_current_result.user = "othertestuser"

        (mock_test_result, mock_page) = mock_page_one
        mock_test_result.results = {"1": [mock_current_result]}

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Current results for this test:"),
            call(str(mock_current_result)),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_additional_result(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test a user submitting an additional result
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        mock_current_result = Mock()
        mock_current_result.user = mock_wiki.username.lower()

        (mock_test_result, mock_page) = mock_page_one
        mock_test_result.results = {"1": [mock_current_result]}

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99, confirm additional result
        mock_get_confirm.side_effect = ["y", "y"]

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Current results for this test:"),
            call(str(mock_current_result)),
            call(ALREADY),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_no_additional_result(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        # Test a user not submitting an additional result
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        mock_current_result = Mock()
        mock_current_result.user = mock_wiki.username.lower()

        # Test results with a result from the current user
        mock_own_test_result = Mock()
        mock_own_test_result.secid = "test 1"
        mock_own_test_result.results = {"1": [mock_current_result]}
        mock_own_test_result.testcase = "test case"

        # Empty test results and mock Page (from fixture)
        (mock_test_result, mock_page) = mock_page_one

        # Return current user's result on first call, empty result on second call
        mock_page.get_resultrows.side_effect = [[mock_own_test_result], [mock_test_result]]

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99, no additional result, confirm release again
        mock_get_confirm.side_effect = ["y", "n", "y"]

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(SystemExit):
            report_results(mock_wiki)

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call(), call(), call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Current results for this test:"),
            call(str(mock_current_result)),
            call(ALREADY),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_add_result_exception(
        self, mock_print, mock_get_result, mock_get_confirm, mock_get_response, mock_page_one
    ):
        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one
        mock_page.add_result.side_effect = Exception("test exception")

        # Quit after get_result
        mock_get_response.return_value = "q"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with pytest.raises(Exception) as excinfo:
            report_results(mock_wiki)

        assert str(excinfo.value) == "test exception"

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        mock_get_response.assert_not_called()

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS")
    def test_report_results_another_section(
        self,
        mock_sections,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_page_one,
    ):
        # Test reporting another result for (s)ection
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        mock_sections.__len__.return_value = 1
        mock_sections.__getitem__.return_value = {"index": "test 1", "line": "test section"}

        # Report another result for this (s)ection
        mock_get_response.return_value = "s"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with patch("relval.report_results.report_results") as mock_report_results:
            # Call the original, unpatched report_result
            original_report_result(mock_wiki)

        mock_report_results.assert_called_with(
            mock_wiki,
            99,
            "test",
            "20220708",
            None,
            {"index": "test 1", "line": "test section"},
            checkver=False,
            dist="Fedora",
        )

        assert relval.report_results.PAGE == mock_page
        assert relval.report_results.SECTIONS == mock_sections

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS")
    def test_report_results_another_page(
        self,
        mock_sections,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_page_one,
    ):
        # Test reporting another result for (p)age
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        mock_sections.__len__.return_value = 1
        mock_sections.__getitem__.return_value = {"index": "test 1", "line": "test section"}

        # Report another result for this (p)age
        mock_get_response.return_value = "p"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with patch("relval.report_results.report_results") as mock_report_results:
            # Call the original, unpatched report_result
            original_report_result(mock_wiki)

        mock_report_results.assert_called_with(
            mock_wiki, 99, "test", "20220708", None, None, checkver=False, dist="Fedora"
        )

        assert relval.report_results.PAGE == mock_page
        assert relval.report_results.SECTIONS == mock_sections

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.get_result")
    @patch("relval.report_results.print")
    @patch("relval.report_results.SECTIONS")
    def test_report_results_another_compose(
        self,
        mock_sections,
        mock_print,
        mock_get_result,
        mock_get_confirm,
        mock_get_response,
        mock_page_one,
    ):
        # Test reporting another result for (c)ompose
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test_result, mock_page) = mock_page_one

        mock_sections.__len__.return_value = 1
        mock_sections.__getitem__.return_value = {"index": "test 1", "line": "test section"}

        # Report another result for this (c)ompose
        mock_get_response.return_value = "c"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        # Result is "pass"
        mock_get_result.return_value = "pass"

        with patch("relval.report_results.report_results") as mock_report_results:
            # Call the original, unpatched report_result
            original_report_result(mock_wiki)

        mock_report_results.assert_called_with(
            mock_wiki, 99, "test", "20220708", None, None, checkver=False, dist="Fedora"
        )

        assert relval.report_results.PAGE is None
        assert relval.report_results.SECTIONS is None

        mock_page.add_result.assert_called_with("pass", mock_test_result, "1")

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response.mock_calls == [
            call(
                ANOTHER,
                "Please choose (s)ection, (p)age, (c)ompose, or (q)uit.",
                ("s", "p", "c", "q"),
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_get_result.mock_calls == [call(mock_wiki.username.lower())]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
            call("Will report pass for test test case in section test section of test page"),
            call("Succeeded!"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.print")
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_back_from_test_type(
        self, mock_print, mock_get_confirm, mock_get_response_list, mock_event_res_pages
    ):
        # Test going back from the "Which test type..." prompt
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.return_value = mock_event

        mock_get_response_list.return_value = "back"

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        with patch("relval.report_results.report_results") as mock_report_results:
            mock_report_results.side_effect = report_results_exit

            with pytest.raises(SystemExit):
                # Call the original, unpatched report_result, the mock will exit
                original_report_result(mock_wiki)

        # "back" makes a recursive call, check it
        mock_report_results.assert_called_with(mock_wiki)

        # Check the state globals
        assert relval.report_results.RESPAGES == [mock_page]
        assert relval.report_results.PAGE == None
        assert relval.report_results.SECTIONS == [{"index": "test 1", "line": "test section"}]

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            )
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.print")
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.PAGE", None)
    @patch(
        "relval.report_results.SECTIONS",
        [
            {"index": "test 1", "line": "test section 1"},
            {"index": "test 1", "line": "test section 2"},
        ],
    )
    def test_report_results_back_from_section(
        self, mock_print, mock_get_confirm, mock_get_response_list, mock_event_res_pages
    ):
        # Test going back from the "Which section..." prompt
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.return_value = mock_event

        # "testtesttype", "back"
        mock_get_response_list.side_effect = [0, "back"]

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        with patch("relval.report_results.report_results") as mock_report_results:
            mock_report_results.side_effect = report_results_exit

            with pytest.raises(SystemExit):
                # Call the original, unpatched report_result, the mock will exit
                original_report_result(mock_wiki)

        # "back" makes a recursive call, check it
        mock_report_results.assert_called_with(
            mock_wiki, 99, "test", "20220708", checkver=False, dist="Fedora"
        )

        # Check the state globals
        assert relval.report_results.RESPAGES == [mock_page]
        assert relval.report_results.PAGE is None
        assert relval.report_results.SECTIONS is None

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            ),
            call(
                "Which section would you like to report a result in? ",
                ["test section 1", "test section 2"],
                backstr="back",
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.print")
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_back_from_test(
        self, mock_print, mock_get_confirm, mock_get_response_list, mock_page_two
    ):
        # Test going back from the "Which test..." prompt
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }

        (mock_test1_result, mock_test2_result, mock_page) = mock_page_two
        mock_page.testtype = "testtesttype"

        mock_event = Mock()
        mock_event.result_pages = [mock_page]

        mock_wiki.get_validation_event.return_value = mock_event

        # "testtesttype", "back"
        mock_get_response_list.side_effect = [0, "back"]

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        with patch("relval.report_results.report_results") as mock_report_results:
            mock_report_results.side_effect = report_results_exit

            with pytest.raises(SystemExit):
                # Call the original, unpatched report_result, the mock will exit
                original_report_result(mock_wiki)

        # "back" makes a recursive call, check it
        mock_report_results.assert_called_with(
            mock_wiki, 99, "test", "20220708", checkver=False, dist="Fedora"
        )

        # Check the state globals
        assert relval.report_results.RESPAGES == [mock_page]
        assert relval.report_results.PAGE == mock_page
        assert relval.report_results.SECTIONS == [{"index": "test 1", "line": "test section"}]

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            ),
            call(
                "Which test would you like to report a result for? ",
                ["test case test name 1", "test case test name 2"],
                backstr="back",
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
        ]

    @patch("relval.report_results.get_response_list")
    @patch("relval.report_results.get_confirm")
    @patch("relval.report_results.print")
    @patch("relval.report_results.RESPAGES", None)
    @patch("relval.report_results.PAGE", None)
    @patch("relval.report_results.SECTIONS", [{"index": "test 1", "line": "test section"}])
    def test_report_results_back_from_environment(
        self, mock_print, mock_get_confirm, mock_get_response_list, mock_event_res_pages
    ):
        # Test going back from the "Which environment..." prompt
        import relval.report_results  # Access to globals pylint:disable=import-outside-toplevel

        original_report_result = reference_wrapper(report_results)

        (mock_test_result, mock_page, mock_event) = mock_event_res_pages

        mock_test_result.results = {"1": None, "2": None}

        mock_wiki = Mock()
        mock_wiki.get_current_compose.return_value = {
            "release": 99,
            "milestone": "test",
            "compose": "",
            "date": "20220708",
        }
        mock_wiki.get_validation_event.return_value = mock_event

        # "testtestype", "back"
        mock_get_response_list.side_effect = [0, "back"]

        # Confirm release 99
        mock_get_confirm.return_value = "y"

        with patch("relval.report_results.report_results") as mock_report_results:
            mock_report_results.side_effect = report_results_exit

            with pytest.raises(SystemExit):
                # Call the original, unpatched report_result, the mock will exit
                original_report_result(mock_wiki)

        # "back" makes a recursive call, check it
        mock_report_results.assert_called_with(
            mock_wiki,
            99,
            "test",
            "20220708",
            None,
            {"index": "test 1", "line": "test section"},
            checkver=False,
            dist="Fedora",
        )

        # Check the state globals
        assert relval.report_results.RESPAGES == [mock_page]
        assert relval.report_results.PAGE == mock_page
        assert relval.report_results.SECTIONS == [{"index": "test 1", "line": "test section"}]

        mock_wiki.get_current_compose.assert_called_with(dist="Fedora")

        assert mock_get_response_list.mock_calls == [
            call(
                "Which test type would you like to report a result for? ",
                ["testtesttype"],
                backstr="back",
            ),
            call(
                "Which environment would you like to report a result for? ",
                ["1", "2"],
                backstr="back",
            ),
        ]

        assert mock_get_confirm.mock_calls == [call()]

        assert mock_print.mock_calls == [
            call("Detecting current compose version..."),
            call("Will report results for compose: Fedora 99 test 20220708"),
        ]
