# fedpkg - a Python library for RPM Packagers
#
# Copyright (C) 2011 Red Hat Inc.
# Author(s): Jesse Keating <jkeating@redhat.com>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
# the full text of the license.

import pyrpkg
import os
import git
import re

from datetime import datetime, timedelta, timezone

# doc/fedpkg_man_page.py uses the 'cli' import
from . import cli  # noqa

try:
    from .bodhi_6 import BodhiClient, UPDATE_TYPES, REQUEST_TYPES, SUGGEST_TYPES
except ImportError:
    # working bodhi-client is not available for el8 (Python3.6)
    BodhiClient = None
from .lookaside import FedoraLookasideCache
from pyrpkg.utils import cached_property

import distro


class Commands(pyrpkg.Commands):

    def __init__(self, *args, **kwargs):
        """Init the object and some configuration details."""

        super(Commands, self).__init__(*args, **kwargs)

        self.source_entry_type = 'bsd'
        # un-block retirement of packages (module retirement is allowed by default)
        if 'rpms' in self.block_retire_ns:
            self.block_retire_ns.remove('rpms')

    def load_user(self):
        """This sets the user attribute, based on the Fedora SSL cert."""
        fedora_upn = os.path.expanduser('~/.fedora.upn')
        if os.path.exists(fedora_upn):
            with open(fedora_upn, 'r') as f:
                self._user = f.read().strip()
        else:
            self.log.debug('Could not get user from .fedora.upn, falling back'
                           ' to default method')
            super(Commands, self).load_user()

    @cached_property
    def lookasidecache(self):
        """A helper to interact with the lookaside cache

        We override this because we need a different download path.
        """
        return FedoraLookasideCache(
            self.lookasidehash, self.lookaside, self.lookaside_cgi,
            attempts=self.lookaside_attempts, delay=self.lookaside_delay)

    def _load_rpmdefines_branch(self, branch, extra_rpmdefines):
        # We only match the top level branch name exactly.
        # Anything else is too dangerous and --dist should be used
        # This regex works until after Fedora 99.
        if re.match(r'f\d\d$', branch):
            self._distval = branch.split('f')[1]
            self._distvar = 'fedora'
            self._disttag = 'fc%s' % self._distval
            self.mockconfig = 'fedora-%s-%s' % (self._distval, self.localarch)
            self.override = 'f%s-override' % self._distval
            self._distunset = 'rhel'
        # Works until RHEL 10
        elif re.match(r'el\d$', branch) or \
                re.match(r'epel\d$', branch):
            self._distval = branch.split('el')[1]
            self._distvar = 'rhel'
            self._disttag = 'el%s' % self._distval
            self.mockconfig = 'epel-%s-%s' % (self._distval, self.localarch)
            self.override = 'epel%s-override' % self._distval
            self._distunset = 'fedora'
        # EPEL 10+ without minor
        elif re.match(r'epel\d{2,}$', branch):
            macros = self._get_build_macros('epel%s-candidate' % branch[4:])
            major, minor = macros['distcore'][3:].split('_')
            self._distval = major
            self._distvar = 'rhel'
            self._disttag = macros['distcore'][1:]
            self.mockconfig = 'epel-%s-%s' % (major, self.localarch)
            self.override = 'epel%s.%s-override' % (major, minor)
            self._distunset = 'fedora'
        # EPEL 10+ with minor
        elif re.match(r'epel\d{2,}\.\d+$', branch):
            major, minor = branch[4:].split('.')
            self._distval = major
            self._distvar = 'rhel'
            self._disttag = 'el%s_%s' % (major, minor)
            self.mockconfig = 'epel-%s.%s-%s' % (major, minor, self.localarch)
            self.override = 'epel%s.%s-override' % (major, minor)
            self._distunset = 'fedora'
        elif re.match(r'epel\d+-next$', branch):
            self._distval = re.search(r'\d+', branch).group(0)
            self._distvar = 'rhel'
            self._disttag = 'el%s.next' % self._distval
            self.mockconfig = 'epel-next-%s-%s' % (self._distval, self.localarch)
            self.override = 'epel%s-next-override' % self._distval
            self._distunset = 'fedora'
        elif re.match(r'olpc\d$', branch):
            self._distval = branch.split('olpc')[1]
            self._distvar = 'olpc'
            self._disttag = 'olpc%s' % self._distval
            self.override = 'dist-olpc%s-override' % self._distval
            self._distunset = 'rhel'
        elif re.match(r'eln$', branch):
            macros = self._get_build_macros('eln-candidate')
            self._distval = macros['eln']
            self._distvar = 'eln'
            self._disttag = 'eln%s' % macros['eln']
            self.mockconfig = 'fedora-eln-%s' % self.localarch
            self.override = 'eln-override'
            self._distunset = 'fedora'
            extra_rpmdefines.extend([
                "--define", "el%s 1" % macros['rhel'],
                "--define", "rhel %s" % macros['rhel'],
            ])
        # rawhide (previously master branch)
        elif re.match(r'(rawhide|main)$', branch):
            self._distval = self._findrawhidebranch()
            self._distvar = 'fedora'
            self._disttag = 'fc%s' % self._distval
            self.mockconfig = 'fedora-rawhide-%s' % self.localarch
            self.override = None
            self._distunset = 'rhel'
        # If we don't match one of the above, punt
        else:
            if self.dist:
                msg = 'Invalid release \'%s\'.' % branch
                raise pyrpkg.rpkgError(msg)
            else:
                return False
        return True

    # Overloaded property loaders
    def load_rpmdefines(self):
        """Populate rpmdefines based on branch data"""

        # Determine runtime environment
        self._runtime_disttag = self._determine_runtime_env()

        extra_rpmdefines = []

        if not self._load_rpmdefines_branch(self.branch_merge, extra_rpmdefines):
            base = self.branch_merge.split('-', 1)
            # Allow branches like f38-staging
            if len(base) > 1:
                self.log.debug('Retrying with branch {}, feature {}'
                               .format(base[0], base[1]))
            if not (len(base) > 1 and base[0].isalnum() and len(base[1]) > 0 and
                    self._load_rpmdefines_branch(base[0], extra_rpmdefines)):
                msg = ('Could not find release from branch name \'%s\'. '
                       'Please specify with --release.' % self.branch_merge)
                raise pyrpkg.rpkgError(msg)

        self._rpmdefines = ["--define", "_sourcedir %s" % self.layout.sourcedir,
                            "--define", "_specdir %s" % self.layout.specdir,
                            "--define", "_builddir %s" % self.layout.builddir,
                            "--define", "_srcrpmdir %s" % self.layout.srcrpmdir,
                            "--define", "_rpmdir %s" % self.layout.rpmdir,
                            "--define", "_rpmfilename %s" % self.layout.rpmfilename,
                            "--define", "dist %%{?distprefix}.%s" % self._disttag,
                            "--define", "%s %s" % (self._distvar, self._distval),
                            "--eval", "%%undefine %s" % self._distunset,
                            "--define", "%s 1" % self._disttag.replace(".", "_")]
        # TODO: consider removing macro "%s 1; it has unknown/dubious functionality"

        # If the runtime is known, and is different from
        # the target, we need to unset the _runtime_disttag
        if self._runtime_disttag:
            # In the case of rhel like runtimes, only the major needs to be different
            if re.match(r'^el\d', self._runtime_disttag):
                if not self._disttag.startswith(self._runtime_disttag):
                    self._rpmdefines.extend(["--eval", "%%undefine %s" %
                                            self._runtime_disttag])
            elif self._disttag != self._runtime_disttag:
                self._rpmdefines.extend(["--eval", "%%undefine %s" %
                                        self._runtime_disttag])

        if extra_rpmdefines:
            self._rpmdefines.extend(extra_rpmdefines)

    def build_target(self, release):
        if release in ('rawhide', 'main'):
            return 'rawhide'
        else:
            return '%s-candidate' % release

    def load_container_build_target(self):
        if self.branch_merge in ('rawhide', 'main'):
            self._container_build_target = 'rawhide-%s-candidate' % self.ns
        else:
            super(Commands, self).load_container_build_target()

    def default_branch_merge(self):
        return "rawhide"

    def _tag2version(self, dest_tag):
        """ get the '26' part of 'f26-foo' string """
        return dest_tag.split('-')[0].replace('f', '')

    def _get_build_macros(self, build_target):
        """Get the koji rpm.build.macros for the given build target"""

        if self._kojisession:
            session = self.kojisession
        else:
            if self.koji_offline:
                # can't continue, the process relies on macros from Koji
                raise pyrpkg.rpkgError(
                    'Offline is active - can\'t get macros for the current release from Koji.')
            session = self.anon_kojisession

        try:
            target = session.getBuildTarget(build_target)
        except Exception as e:
            self.log.error(str(e))
            raise pyrpkg.rpkgError(
                'Cannot access Koji to get build information for {}'.format(build_target))

        buildcfg = session.getBuildConfig(target['build_tag_name'])

        macros = {}
        for extra in buildcfg['extra']:
            if extra.startswith('rpm.macro.'):
                macroname = extra[len('rpm.macro.'):]
                macros[macroname] = buildcfg['extra'][extra]

        return macros

    def _findrawhidebranch(self):
        """Find the right "fedora" for rawhide (previously master branch)"""

        # If we already have a koji session, just get data from the source
        if self._kojisession:
            rawhidetarget = self.kojisession.getBuildTarget('rawhide')
            return self._tag2version(rawhidetarget['dest_tag_name'])

        # Try connect Koji once more, this time with anonymous session.
        try:
            rawhidetarget = self.anon_kojisession.getBuildTarget('rawhide')
        except Exception:
            # We couldn't hit Koji. Continue, because fedpkg may work offline.
            self.log.debug('Unable to query Koji to find rawhide target. Continue offline.')
        else:
            return self._tag2version(rawhidetarget['dest_tag_name'])

        # Create a list of "fedoras"
        fedoras = []

        # Create a regex to find branches that exactly match f##.  Should not
        # catch branches such as f14-foobar
        branchre = r'f\d\d$'

        # Find the repo refs
        for ref in self.repo.refs:
            # Only find the remote refs
            if isinstance(ref, git.RemoteReference):
                # Search for branch name by splitting off the remote
                # part of the ref name and returning the rest.  This may
                # fail if somebody names a remote with / in the name...
                if re.match(branchre, ref.name.split('/', 1)[1]):
                    # Add just the simple f## part to the list
                    fedoras.append(ref.name.split('/')[1])
        if fedoras:
            # Sort the list
            fedoras.sort()
            # Start with the last item, strip the f, add 1, return it.
            return int(fedoras[-1].strip('f')) + 1
        else:
            raise pyrpkg.rpkgError('Unable to find rawhide target')

    def _getcommitmessage(self):
        """Returns the latest commit message on the current branch"""

        try:
            commitmsg = next(self.repo.iter_commits()).message
        except TypeError:
            commitmsg = ""

        return commitmsg

    def _isrpmautospec(self):
        """Returns whether the spec file uses rpmautospec features"""

        return self._uses_rpmautospec

    def _determine_runtime_env(self):
        """Need to know what the runtime env is, so we can unset anything
           conflicting
        """
        if distro.id() == 'rhel' or 'rhel' in distro.like().split():
            return 'el%s' % distro.major_version()
        if distro.id() == 'fedora':
            os_release_info = distro.os_release_info()
            if 'variant' in os_release_info and os_release_info['variant'] == 'ELN':
                macros = self._get_build_macros('eln-candidate')
                return 'eln%s' % macros['eln']
            return 'fc%s' % distro.major_version()
        return None

    def check_inheritance(self, build_target, dest_tag):
        """Disable check inheritance

        Tag inheritance check is not required in Fedora when make chain build
        in Koji.
        """

    def construct_build_url(self, *args, **kwargs):
        """Override build URL for Fedora Koji build

        In Fedora Koji, anonymous URL should have prefix "git+https://"
        """
        url = super(Commands, self).construct_build_url(*args, **kwargs)
        return 'git+{0}'.format(url)

    def update(self, bodhi_config, template='bodhi.template', bugs=[]):
        """Submit an update to bodhi using the provided template."""
        bodhi = BodhiClient(username=self.user,
                            staging=bodhi_config['staging'])

        update_details = bodhi.parse_file(template)

        for detail in update_details:
            if not detail['type']:
                raise ValueError(
                    'Missing update type, which is required to create update.')
            if detail['type'] not in UPDATE_TYPES:
                raise ValueError(
                    'Incorrect update type {0}'.format(detail['type']))
            if detail['request'] not in REQUEST_TYPES:
                raise ValueError(
                    'Incorrect request type {0}'.format(detail['request']))
            if detail['suggest'] not in SUGGEST_TYPES:
                raise ValueError(
                    'Incorrect suggest type {0}'.format(detail['suggest']))

            try:
                self.log.info(bodhi.update_str(bodhi.save(**detail), minimal=False))
            # Only because tests do not return a valid bodhi.save value
            except TypeError:
                pass

    def create_buildroot_override(self, bodhi_config, build, duration,
                                  notes=''):
        bodhi = BodhiClient(username=self.user,
                            staging=bodhi_config['staging'])
        result = bodhi.list_overrides(builds=build)
        if result['total'] == 0:
            try:
                self.log.debug(
                    'Create override in %s: nvr=%s, duration=%s, notes="%s"',
                    'staging Bodhi' if bodhi_config['staging'] else 'Bodhi',
                    build, duration, notes)
                override = bodhi.save_override(
                    nvr=build, duration=duration, notes=notes)
            except Exception as e:
                self.log.error(str(e))
                raise pyrpkg.rpkgError('Cannot create override.')
            else:
                self.log.info(bodhi.override_str(override, minimal=False))
        else:
            override = result['overrides'][0]
            expiration_date = datetime.strptime(override['expiration_date'],
                                                '%Y-%m-%d %H:%M:%S')
            if expiration_date < datetime.now(timezone.utc).replace(tzinfo=None):
                self.log.info(
                    'Buildroot override for %s exists and is expired. Consider'
                    ' using command `override extend` to extend duration.',
                    build)
            else:
                self.log.info('Buildroot override for %s already exists and '
                              'not expired.', build)

    def extend_buildroot_override(self, bodhi_config, build, duration):
        bodhi = BodhiClient(username=self.user,
                            staging=bodhi_config['staging'])
        result = bodhi.list_overrides(builds=build)

        if result['total'] == 0:
            self.log.info('No buildroot override for build %s', build)
            return

        override = result['overrides'][0]
        expiration_date = datetime.strptime(override['expiration_date'],
                                            '%Y-%m-%d %H:%M:%S')
        utcnow = datetime.now(timezone.utc).replace(tzinfo=None)

        # bodhi-client binding API save_override calculates expiration
        # date by adding duration to datetime.utcnow
        # This comparison should use utcnow as well.
        if expiration_date < utcnow:
            self.log.debug('Buildroot override is expired on %s',
                           override['expiration_date'])
            self.log.debug('Extend expiration date from today in UTC.')
            base_date = utcnow
        else:
            self.log.debug(
                'Extend expiration date from future expiration date.')
            base_date = expiration_date

        if isinstance(duration, datetime):
            if duration < utcnow:
                raise pyrpkg.rpkgError(
                    'At least, specified expiration date {0} should be '
                    'future date.'.format(duration.strftime('%Y-%m-%d')))
            if duration < base_date:
                self.log.warning(
                    'Expiration date %s to be set is before override current'
                    ' expiration date %s',
                    duration, base_date)
            # Keep time unchanged
            new_expiration_date = datetime(
                year=duration.year,
                month=duration.month,
                day=duration.day,
                hour=base_date.hour,
                minute=base_date.minute,
                second=base_date.second)
        else:
            new_expiration_date = base_date + timedelta(days=duration)

        try:
            self.log.debug('Extend override expiration date to %s',
                           new_expiration_date)
            override = bodhi.extend_override(override, new_expiration_date)
        except Exception as e:
            self.log.error('Cannot extend override expiration.')
            raise pyrpkg.rpkgError(str(e))
        else:
            self.log.info(bodhi.override_str(override, minimal=False))

    @staticmethod
    def use_local_mock_config(_, force_local):
        if force_local is not None:
            return force_local
        return True


if __name__ == "__main__":
    from fedpkg.__main__ import main
    main()
