# -*- coding: utf-8 -*-
# cli.py - a cli client class module for fedpkg
#
# Copyright (C) 2011 Red Hat Inc.
# Author(s): Jesse Keating <jkeating@redhat.com>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
# the full text of the license.

from __future__ import print_function

import argparse
import io
import itertools
import json
import os
import re
import shutil
import textwrap
from datetime import datetime


# Use deprecated pkg_resources if importlib isn't available (python 3.6)
try:
    from importlib.metadata import distribution
except ImportError:
    from pkg_resources import get_distribution as distribution
# Use deprecated pkg_resources if packaging library isn't available (python 3.6)
try:
    from packaging.version import parse as parse_version
except ImportError:
    from pkg_resources import parse_version
from pyrpkg import rpkgError
from pyrpkg.cli import cliClient
import configparser
from urllib.parse import urlparse

from fedpkg.bugzilla import BugzillaClient
from fedpkg.completers import (build_arches, distgit_branches, fedpkg_packages,
                               list_targets)
from fedpkg.utils import (assert_new_tests_repo, assert_valid_epel_package,
                          config_get_safely, disable_monitoring, do_add_remote,
                          do_fork, expand_release, get_dist_git_url,
                          get_fedora_release_state, get_pagure_branches,
                          get_release_branches, get_stream_branches, is_epel,
                          new_pagure_issue, sl_list_to_dict, verify_sls,
                          does_package_exist_in_anitya, get_last_commit_date)

RELEASE_BRANCH_REGEX = r'^(f\d+|el\d+|eln|epel\d+|epel\d+\.\d+)$'
BUGZILLA_URL_REGEX = r"https:\/\/bugzilla\.redhat\.com\/show_bug\.cgi\?id=(\d{7})"
LOCAL_PACKAGE_CONFIG = 'package.cfg'

BODHI_TEMPLATE = """\
[ %(nvr)s ]

# bugfix, security, enhancement, newpackage, unspecified (required)
type=%(type_)s

# testing, stable
request=%(request)s

# Bug numbers: 1234,9876
bugs=%(bugs)s

# Severity: low, medium, high, urgent
# This is required for security updates.
severity=%(severity)s

display_name=

%(changelog)s
# Here is where you give an explanation of your update.
# Content can span multiple lines, as long as they are indented deeper than
# the first line. For example,
# notes=first line
#     second line
#     and so on
notes=%(descr)s

# Enable request automation based on the stable/unstable karma thresholds
autokarma=%(autokarma)s
stable_karma=%(stable_karma)s
unstable_karma=%(unstable_karma)s

# Automatically close bugs when this marked as stable
close_bugs=%(close_bugs)s

# Suggest that users performs one of the following actions after the update:
# unspecified, restart, logout
# The default value is unspecified
suggest=%(suggest)s

# A boolean to require that all of the bugs in your update have been confirmed by testers.
require_bugs=%(require_bugs)s

# A boolean to require that this update passes all test cases before reaching stable.
require_testcases=%(require_testcases)s
"""


def check_bodhi_version():
    # Use deprecated pkg_resources if importlib isn't available (python 3.6)
    bodhi_version = distribution('bodhi-client').version
    if parse_version(bodhi_version) < parse_version("6.0.0"):
        raise rpkgError('bodhi-client < 6.0.0 is not supported.')


class fedpkgClient(cliClient):
    def __init__(self, config, name=None):
        self.DEFAULT_CLI_NAME = 'fedpkg'
        super(fedpkgClient, self).setup_completers()
        self.setup_completers()
        super(fedpkgClient, self).__init__(config, name)
        self.setup_fed_subparsers()

    def setup_argparser(self):
        super(fedpkgClient, self).setup_argparser()

        # This line is added here so that it shows up with the "--help" option,
        # but it isn't used for anything else
        self.parser.add_argument(
            '--user-config', help='Specify a user config file to use')
        opt_release = self.parser._option_string_actions['--release']
        opt_release.help = 'Override the discovered release, e.g. f25, which has to match ' \
                           'the remote branch name created in package repository. ' \
                           'Particularly, use rawhide/main branch to build RPMs for rawhide.'

    def setup_fed_subparsers(self):
        """Register the fedora specific targets"""

        self.register_releases_info()
        self.register_update()
        self.register_request_repo()
        self.register_request_tests_repo()
        self.register_request_branch()
        self.register_do_fork()
        self.register_override()
        self.register_set_distgit_token()
        self.register_set_pagure_token()
        self.register_do_disable_monitoring()
        self.register_request_unretirement()

    def setup_completers(self):
        """
        Set specific argument completers for fedpkg. Structure, where
        are these assignments (name -> method) stored, is in the parent
        class and have to be filled before __init__ (containing argument
        parser definitions) is called there.
        """
        cliClient.set_completer("build_arches", build_arches)
        cliClient.set_completer("list_targets", list_targets)
        cliClient.set_completer("packages", fedpkg_packages)
        cliClient.set_completer("branches", distgit_branches)

    # Target registry goes here
    def register_update(self):
        description = textwrap.dedent('''
            This will create a bodhi update request for the current package n-v-r.

            There are two ways to specify update details. Without any argument from command
            line, either update type or notes is omitted, a template editor will be shown
            and let you edit the detail information interactively.

            Alternatively, you could specify argument from command line to create an update
            directly, for example:

                {0} update --type bugfix --notes 'Rebuilt' --bugs 1000 1002

            When all lines in template editor are commented out or deleted, the creation
            process is aborted. If the template keeps unchanged, {0} continues on creating
            update. That gives user a chance to confirm the auto-generated notes from
            change log if option --notes is omitted.
        '''.format(self.name))

        update_parser = self.subparsers.add_parser(
            'update',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help='Submit last build as update',
            description=description,
        )

        def validate_stable_karma(value):
            error = argparse.ArgumentTypeError(
                'Stable karma must be an integer which is greater than zero.')
            try:
                karma = int(value)
            except ValueError:
                raise error
            if karma <= 0:
                raise error
            return karma

        def validate_unstable_karma(value):
            error = argparse.ArgumentTypeError(
                'Unstable karma must be an integer which is less than zero.')
            try:
                karma = int(value)
            except ValueError:
                raise error
            if karma >= 0:
                raise error
            return karma

        def validate_bugs(value):
            if not value.isdigit():
                raise argparse.ArgumentTypeError(
                    'Invalid bug {0}. It should be an integer.'.format(value))
            return value

        update_parser.add_argument(
            '--type',
            choices=['bugfix', 'security', 'enhancement', 'newpackage', 'unspecified'],
            dest='update_type',
            help='Update type. Template editor will be shown if type is '
                 'omitted.')
        update_parser.add_argument(
            '--request',
            choices=['testing', 'stable'],
            default='testing',
            help='Requested repository.')
        update_parser.add_argument(
            '--bugs',
            nargs='+',
            type=validate_bugs,
            help='Bug numbers. If omitted, bug numbers will be extracted from'
                 ' change logs.')
        update_parser.add_argument(
            '--notes',
            help='Update description. Multiple lines of notes could be '
                 'specified. If omitted, template editor will be shown.')
        update_parser.add_argument(
            '--disable-autokarma',
            action='store_false',
            default=True,
            dest='autokarma',
            help='Karma automatism is enabled by default. Use this option to '
                 'disable that.')
        update_parser.add_argument(
            '--stable-karma',
            type=validate_stable_karma,
            metavar='KARMA',
            default=3,
            help='Stable karma. Default is 3.')
        update_parser.add_argument(
            '--unstable-karma',
            type=validate_unstable_karma,
            metavar='KARMA',
            default=-3,
            help='Unstable karma. Default is -3.')
        update_parser.add_argument(
            '--not-close-bugs',
            action='store_false',
            default=True,
            dest='close_bugs',
            help='By default, update will be created by enabling to close bugs'
                 ' automatically. If this is what you do not want, use this '
                 'option to disable the default behavior.')
        update_parser.add_argument(
            '--no-require-bugs',
            action='store_false',
            default=True,
            dest='require_bugs',
            help='Disables the requirement that all of the bugs in your update '
                 'have been confirmed by testers. Default is True.')
        update_parser.add_argument(
            '--no-require-testcases',
            action='store_false',
            default=True,
            dest='require_testcases',
            help='Disables the requirement that this update passes all test cases '
                 'before reaching stable. Default is True.')
        update_parser.add_argument(
            '--severity',
            choices=['unspecified', 'low', 'medium', 'high', 'urgent'],
            default='unspecified',
            help='Severity - required for security updates')

        group = update_parser.add_mutually_exclusive_group()
        group.add_argument(
            '--suggest-reboot',
            action='store_true',
            default=False,
            dest='suggest_reboot',
            help='Suggest user to reboot after update. Default is False.')
        group.add_argument(
            '--suggest-logout',
            action='store_true',
            default=False,
            dest='suggest_logout',
            help='Suggest user to logout after update. Default is False.')

        update_parser.set_defaults(command=self.update)

    def get_distgit_namespaces(self):
        dg_namespaced = self._get_bool_opt('distgit_namespaced')
        if dg_namespaced and self.config.has_option(
                self.name, 'distgit_namespaces'):
            return self.config.get(self.name, 'distgit_namespaces').split()
        else:
            return None

    def _common_api_key_description(self):
        pagure_section = '{0}.pagure'.format(self.name)
        pagure_url = config_get_safely(self.config, pagure_section, 'url')
        pagure_url_parsed = urlparse(pagure_url).netloc

        return '''
            Before the operation, you need to generate a pagure.io API token at:
                https://{1}/settings/token/new

                ACL required:
                    "Create a new ticket"

            Update your token with the following command:
                {0} set-pagure-token

            Command saves token to {0} config file:
                ~/.config/rpkg/{0}.conf

            For example:
                [{0}.pagure]
                token = <api_key_here>
            '''.format(self.name, pagure_url_parsed)

    def register_request_repo(self):
        help_msg = 'Request a new dist-git repository'

        description = textwrap.dedent('''
            Request a new dist-git repository

            {0}

            Below is a basic example of the command to request a dist-git repository for
            the package foo:

                fedpkg request-repo foo 1234

            Another example to request a module foo:

                fedpkg request-repo --namespace modules foo

            If you would like to create a project in release-monitoring
            when requesting a new dist-git repository, use the following
            command with mandatory arguments.
            The full list of arguments and their values is in the doc:
            https://release-monitoring.org/static/docs/user-guide.html:

                fedpkg request-repo foo 1234 --monitor \\
                    --backend custom --upstreamurl "project's url" \\
                    --project-name bar
        '''.format(self._common_api_key_description()))

        request_repo_parser = self.subparsers.add_parser(
            'request-repo',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help=help_msg,
            description=description)
        request_repo_parser.add_argument(
            'name',
            help='Repository name to request.')
        request_repo_parser.add_argument(
            'bug', nargs='?', type=int,
            help='Bugzilla bug ID of the package review request. '
                 'Not required for requesting a module repository')
        request_repo_parser.add_argument(
            '--namespace',
            required=False,
            default='rpms',
            choices=self.get_distgit_namespaces(),
            dest='new_repo_namespace',
            help='Namespace of repository. If omitted, default to rpms.')
        request_repo_parser.add_argument(
            '--description', '-d', help='The repo\'s description in dist-git')
        monitoring_choices = [
            'no-monitoring', 'monitoring', 'monitoring-with-scratch']
        request_repo_parser.add_argument(
            '--monitor', '-m', help='The Anitya monitoring type for the repo',
            choices=monitoring_choices, default=monitoring_choices[1])
        request_repo_parser.add_argument(
            '--upstreamurl', '-u',
            help='The upstream URL of the project')
        request_repo_parser.add_argument(
            '--summary', '-s',
            help='Override the package\'s summary from the Bugzilla bug')
        request_repo_parser.add_argument(
            '--exception', action='store_true',
            help='The package is an exception to the regular package review '
                 'process (specifically, it does not require a Bugzilla bug)')
        request_repo_parser.add_argument(
            '--project-name', help='The package\'s project name', required=False
        )
        backend_choices = [
            "BitBucket", "Cgit", "CPAN", "CRAN", "crates.io", "Debian project",
            "Drupal6", "Drupal7", "folder", "Freshmeat", "GitHub",
            "GitLab", "Gitea", "GNOME", "GNU Project", "Gogs",
            "Google code", "Hackage", "Launchpad", "Maven Central", "npmjs", "Packagist",
            "pagure", "PEAR", "PECL", "PyPI", "Rubygems", "Sourceforge",
            "Sourceforge (git)", "SourceHut", "Stackage", "custom",
        ]
        request_repo_parser.add_argument(
            '--backend', choices=backend_choices,
            help='The package\'s project backend', required=False
        )
        request_repo_parser.add_argument(
            '--no-initial-commit',
            action='store_true',
            help='Do not include an initial commit in the repository.')
        packit_onboarding_choices = ['no', 'pull-request', 'push']
        request_repo_parser.add_argument(
            '--onboard-packit',
            help='Whether to generate a default Packit configuration '
                 'for the package and open a pull request against or push '
                 'directly to the newly created dist-git repository',
            choices=packit_onboarding_choices, default=packit_onboarding_choices[0])
        request_repo_parser.set_defaults(command=self.request_repo)

    def register_request_tests_repo(self):
        help_msg = 'Request a new tests dist-git repository'
        anongiturl = self.config.get(
            self.name, 'anongiturl', vars={'repo': 'any', 'module': 'any'}
        )
        description = textwrap.dedent('''
            Request a new dist-git repository in tests shared namespace

                {1}/projects/tests/*

            For more information about tests shared namespace see

                https://docs.fedoraproject.org/en-US/ci/share-test-code

            {2}

            Below is a basic example of the command to request a dist-git repository for
            the space tests/foo:

                {0} request-tests-repo foo "Description of the repository"

            Note that the space name needs to reflect the intent of the tests and will
            undergo a manual review.
        '''.format(
            self.name,
            get_dist_git_url(anongiturl),
            self._common_api_key_description(),
            )
        )

        request_tests_repo_parser = self.subparsers.add_parser(
            'request-tests-repo',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help=help_msg,
            description=description)
        request_tests_repo_parser.add_argument(
            'name',
            help='Repository name to request.')
        request_tests_repo_parser.add_argument(
            'description',
            help='Description of the tests repository')
        request_tests_repo_parser.add_argument(
            '--bug', type=int,
            help='Bugzilla bug ID of the package review request.')
        request_tests_repo_parser.set_defaults(command=self.request_tests_repo)

    def register_request_branch(self):
        help_msg = 'Request a new dist-git branch'
        description = textwrap.dedent('''
            Request a new dist-git branch

            {1}

            Branch name could be one of current active Fedora and EPEL releases. Use
            command ``{0} releases-info`` to get release names that can be used to request
            a branch.

            Below are various examples of requesting a dist-git branch.

            Request a branch inside a cloned package repository:

                {0} request-branch f27

            Request a branch outside package repository, which could apply to cases of
            requested repository has not been approved and created, or just not change
            directory to package repository:

                {0} request-branch --repo foo f27

            Request a branch with service level tied to the branch. In this case branch
            argument has to be before --sl argument, because --sl allows multiple values.

                {0} request-branch branch_name --sl bug_fixes:2020-06-01 rawhide:2019-12-01
        '''.format(self.name, self._common_api_key_description()))

        request_branch_parser = self.subparsers.add_parser(
            'request-branch',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help=help_msg,
            description=description)
        request_branch_parser.add_argument(
            'branch', nargs='?', help='The branch to request.')
        request_branch_parser.add_argument(
            '--repo',
            required=False,
            dest='repo_name_for_branch',
            metavar='NAME',
            help='Repository name the new branch is requested for.'
        )
        request_branch_parser.add_argument(
            '--namespace',
            required=False,
            dest='repo_ns_for_branch',
            choices=self.get_distgit_namespaces(),
            help=('Namespace where the repository specified with --repo '
                  'exists. If omitted, defaults to rpms.')
        )
        request_branch_parser.add_argument(
            '--sl', nargs='*',
            help=('The service levels (SLs) tied to the branch. This must be '
                  'in the format of "sl_name:2020-12-01". This is only for '
                  'non-release branches. You may provide more than one by '
                  'separating each SL with a space. When the argument is used, '
                  'branch argument has to be placed before --sl.')
        )
        request_branch_parser.add_argument(
            '--no-git-branch', default=False, action='store_true',
            help='Don\'t create the branch in git but still create it in PDC'
        )
        request_branch_parser.add_argument(
            '--no-auto-module', default=False, action='store_true',
            help='If requesting an rpm arbitrary branch, do not '
            'also request a new matching module.  See '
            'https://pagure.io/fedrepo_req/issue/129'
        )
        request_branch_parser.add_argument(
            '--all-releases', default=False, action='store_true',
            help='Make a new branch request for every active Fedora release'
        )
        request_branch_parser.set_defaults(command=self.request_branch)

    def register_request_unretirement(self):
        help_msg = "Request an unretirement of the package branch"
        description = textwrap.dedent("""
            Request an unretirement of the package branch

            {1}

            The branch name could be one of the current active Fedora and EPEL releases. Use
            the command ``{0} releases-info`` to get release names that can be used to request
            a branch.

            Below are various examples of requesting an unretirement of a package branch.

            Request an unretirement of the `rawhide` branch inside the cloned repository.
            It will use the discovered username. Use `--user USER` to override it.
            Also, if the branch was retired more than 8 weeks ago, it will ask for an open
            Bugzilla ticket.

                {0} request-unretirement

            Request an unretirement of few branches:

                {0} request-unretirement -b rawhide f40

            Request an unretirement of branch that was retired more than 8 weeks ago
            and requires bugzilla review to proceed:

                {0} request-unretirement --bz_bug_id BUG_ID

            Request an unretirement of package with different namespace than `rpms`:

                {0} request-unretirement --namespace test

            Request an unretirement outside package repository:

                {0} request-unretirement --repo name_of_package
        """.format(self.name, self._common_api_key_description()))
        request_unretirement_parser = self.subparsers.add_parser(
            "request-unretirement",
            formatter_class=argparse.RawTextHelpFormatter,
            help=help_msg,
            description=description,
        )
        request_unretirement_parser.add_argument(
            '--repo',
            required=False,
            dest='repo_name',
            metavar='NAME',
            help='Repository name to unretire some branches in.'
        )
        request_unretirement_parser.add_argument(
            '--namespace',
            required=False,
            dest='repo_ns_name',
            metavar='NS',
            default='rpms',
            help='Repository namespace name, as a default use `rpm`.',
        )
        request_unretirement_parser.add_argument(
            "--bz_bug_id",
            required=False,
            dest="bz_bug_id",
            metavar="BUG_ID",
            default=None,
            help="Bugzilla bug ID with re-review."
        )
        request_unretirement_parser.add_argument(
            "-b", "--branches",
            required=False,
            nargs='+',
            dest="branches",
            help="Comma-separated list of branches for unretirement."
        )
        request_unretirement_parser.set_defaults(command=self.request_unretirement)

    def register_do_fork(self):
        help_msg = 'Create a new fork of the current repository'
        distgit_section = '{0}.distgit'.format(self.name)
        distgit_api_base_url = config_get_safely(self.config, distgit_section, "apibaseurl")

        description = textwrap.dedent('''
            Create a new fork of the current repository

            Before the operation, you need to generate a pagure.io API token at:
                https://{1}/settings/token/new

            ACL required:
                "Fork a project"

            Update your token with the following command:
                fedpkg set-distgit-token

            Command saves token to fedpkg config file:
                ~/.config/rpkg/{0}.conf

            For example:
                [{0}.distgit]
                token = <api_key_here>

            Below is a basic example of the command to fork a current repository:

                {0} fork

            Operation requires username (FAS_ID). by default, current logged
            username is taken. It could be overridden by reusing an argument:

                {0} --user FAS_ID fork
        '''.format(self.name, urlparse(distgit_api_base_url).netloc))

        fork_parser = self.subparsers.add_parser(
            'fork',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help=help_msg,
            description=description)
        fork_parser.add_argument(
            '--anonymous', '-a', action='store_true',
            help='In this mode, it is possible to fork the repository which was cloned '
                 '"anonymously" (fedpkg clone -a). The forked repository uses https '
                 'protocol instead ssh.')
        fork_parser.set_defaults(command=self.do_distgit_fork)

    def register_releases_info(self):
        help_msg = 'Print Fedora or EPEL current active releases'
        parser = self.subparsers.add_parser(
            'releases-info',
            help=help_msg,
            description=help_msg)

        group = parser.add_mutually_exclusive_group()
        group.add_argument(
            '-e', '--epel',
            action='store_true',
            dest='show_epel_only',
            help='Only show EPEL releases.')
        group.add_argument(
            '-f', '--fedora',
            action='store_true',
            dest='show_fedora_only',
            help='Only show Fedora active releases.')
        group.add_argument(
            '-j', '--join',
            action='store_true',
            help='Show all releases in one line separated by a space.')

        parser.set_defaults(command=self.show_releases_info)

    def register_set_distgit_token(self):
        distgit_section = '{0}.distgit'.format(self.name)
        distgit_api_base_url = config_get_safely(self.config, distgit_section, "apibaseurl")
        help_msg = \
            'Updates the fedpkg.distgit API token in ~/.config/rpkg/{0}.conf file.\n\n\
            Tokens are of length 64 and contain only uppercase and numerical values.\n\
            The new API token can be generated at: \n\
            https://{1}/settings/token/new'\
            .format(self.name, urlparse(distgit_api_base_url).netloc)

        parser = self.subparsers.add_parser(
            'set-distgit-token',
            help=help_msg,
            description=help_msg)

        parser.set_defaults(command=self.set_distgit_token)

    def register_set_pagure_token(self):
        pagure_section = '{0}.pagure'.format(self.name)
        pagure_url = config_get_safely(self.config, pagure_section, 'url')
        help_msg = \
            'Updates the fedpkg.pagure API token in ~/.config/rpkg/{0}.conf file.\n\n\
            Tokens are of length 64 and contain only uppercase and numerical values.\n\
            The new API token. Can be generated at: \n\
            https://{1}/settings/token/new'\
            .format(self.name, urlparse(pagure_url).netloc)

        parser = self.subparsers.add_parser(
            'set-pagure-token',
            help=help_msg,
            description=help_msg)

        parser.set_defaults(command=self.set_pagure_token)

    def register_override(self):
        """Register command line parser for subcommand override

        .. versionadded:: 1.34
        """

        def validate_duration(value):
            try:
                duration = int(value)
            except ValueError:
                raise argparse.ArgumentTypeError('duration must be an integer.')
            if duration > 0:
                return duration
            raise argparse.ArgumentTypeError(
                'override should have 1 day to exist at least.')

        def validate_extend_duration(value):
            if value.isdigit():
                return validate_duration(value)
            match = re.match(r'(\d{4})-(\d{1,2})-(\d{1,2})', value)
            if not match:
                raise argparse.ArgumentTypeError(
                    'Invalid expiration date. Valid format: yyyy-mm-dd.')
            y, m, d = match.groups()
            return datetime(year=int(y), month=int(m), day=int(d))

        override_parser = self.subparsers.add_parser(
            'override',
            help='Manage buildroot overrides')
        override_subparser = override_parser.add_subparsers(
            description='Commands on override')

        create_parser = override_subparser.add_parser(
            'create',
            help='Create buildroot override from build',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=textwrap.dedent('''
                Create a buildroot override from build guessed from current release branch or
                specified explicitly.

                Examples:

                Create a buildroot override from build guessed from release branch. Note that,
                command must run inside a package repository.

                    {0} switch-branch f28
                    {0} override create --duration 5

                Create for a specified build:

                    {0} override create --duration 5 package-1.0-1.fc28
            '''.format(self.name)))
        create_parser.add_argument(
            '--duration',
            type=validate_duration,
            default=7,
            help='Number of days the override should exist. If omitted, '
                 'default to 7 days.')
        create_parser.add_argument(
            '--notes',
            default='No explanation given...',
            help='Optional notes on why this override is in place.')
        create_parser.add_argument(
            'NVR',
            nargs='?',
            help='Create override from this build. If omitted, build will be'
                 ' guessed from current release branch.')
        create_parser.set_defaults(command=self.create_buildroot_override)

        extend_parser = override_subparser.add_parser(
            'extend',
            help='Extend buildroot override expiration',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=textwrap.dedent('''
                Extend buildroot override expiration.

                An override expiration date could be extended by number of days or a specific
                date. If override is expired, expiration date will be extended from the date
                of today, otherwise from the expiration date.

                Command extend accepts an optional build NVR to find out its override in
                advance. If there is no such an override created previously, please use
                `override create` to create one. If build NVR is omitted, command extend must
                run inside a package repository and build will be guessed from current release
                branch.

                Examples:

                1. To give 2 days to override for build somepkg-0.2-1.fc28

                    {0} override extend 2 somepkg-0.2-1.fc28

                2. To extend expiration date to 2018-7-1

                    cd /path/to/somepkg
                    {0} switch-branch f28
                    {0} override extend 2018-7-1
            '''.format(self.name)))
        extend_parser.add_argument(
            'duration',
            type=validate_extend_duration,
            help='Number of days to extend the expiration date, or set the '
                 'expiration date directly. Valid date format: yyyy-mm-dd.')
        extend_parser.add_argument(
            'NVR',
            nargs='?',
            help='Buildroot override expiration for this build will be '
                 'extended. If omitted, build will be guessed from current '
                 'release branch.')
        extend_parser.set_defaults(command=self.extend_buildroot_override)

    def register_build(self):
        super(fedpkgClient, self).register_build()

        build_parser = self.subparsers.choices['build']
        build_parser.formatter_class = argparse.RawDescriptionHelpFormatter
        build_parser.description = textwrap.dedent('''
            {0}

            fedpkg is also able to submit multiple builds to Koji at once from stream
            branch based on a local config, which is inside the repository. The config file
            is named package.cfg in INI format. For example,

                [koji]
                targets = rawhide fedora epel7

            You only need to put Fedora releases and EPEL in option targets and fedpkg will
            convert it to proper Koji build target for submitting builds. Beside regular
            release names, option targets accepts two shortcut names as well, fedora and
            epel, as you can see in the above example. Name fedora stands for current
            active Fedora releases, and epel stands for the active EPEL releases, which are
            el6 and epel7 currently.

            Note that the config file is a branch specific file. That means you could
            create package.cfg for each stream branch separately to indicate on which
            targets to build the package for a particular stream.
        ''').format('\n'.join(textwrap.wrap(build_parser.description)))

    def register_do_disable_monitoring(self):
        help_msg = 'Disable monitoring of the current repository'
        distgit_section = '{0}.distgit'.format(self.name)
        distgit_api_base_url = config_get_safely(self.config, distgit_section, "apibaseurl")

        description = textwrap.dedent('''
            Disable monitoring of the current (retired) repository.

            Currently, it doesn't run automatically after the retire
            operation - only manually.

            No arguments are required but a valid token is needed.
            Before the operation, you need to generate a pagure.io API token at:
                https://{1}/settings/token/new

            ACL required:
                "Modify an existing project"

            Update your token with the following command:
                fedpkg set-distgit-token

            Command saves token to fedpkg config file:
                ~/.config/rpkg/{0}.conf

            For example:
                [{0}.distgit]
                token = <api_key_here>

            This operation can be run repeatedly till it succeeds.
        '''.format(self.name, urlparse(distgit_api_base_url).netloc))

        disable_monitoring_parser = self.subparsers.add_parser(
            'disable-monitoring',
            formatter_class=argparse.RawDescriptionHelpFormatter,
            help=help_msg,
            description=description)
        disable_monitoring_parser.set_defaults(command=self.do_disable_monitoring)

    def register_retire(self):
        """Extends the retire's parent description"""
        super(fedpkgClient, self).register_retire()

        retire_parser = self.subparsers.choices['retire']
        retire_parser.formatter_class = argparse.RawDescriptionHelpFormatter
        retire_parser.description = textwrap.dedent('''
            {0}

            After a successful package retirement, there is an optional subsequent
            step of executing a command:

                {1} disable-monitoring

            The `disable-monitoring` command is executed manually if desired.

        ''').format('\n'.join(textwrap.wrap(retire_parser.description)), self.name)

    # Target functions go here
    def _format_update_clog(self, clog):
        ''' Format clog for the update template. '''
        lines = [ln for ln in clog.split('\n') if ln]
        if len(lines) == 0:
            return "- Rebuilt.", ""
        elif len(lines) == 1:
            return lines[0], ""
        log = ["# Changelog:"]
        log.append('# - ' + lines[0])
        for ln in lines[1:]:
            log.append('# ' + ln)
        log.append('#')
        return lines[0], "\n".join(log)

    def _get_bodhi_config(self):
        try:
            section = '%s.bodhi' % self.name
            return {
                'staging': self.config.getboolean(section, 'staging'),
            }
        except (ValueError, configparser.NoOptionError, configparser.NoSectionError) as e:
            self.log.error(str(e))
            raise rpkgError('Could not get bodhi options. It seems configuration is changed. '
                            'Please try to reinstall %s or consult developers to see what '
                            'is wrong with it.' % self.name)

    @staticmethod
    def is_update_aborted(template_file):
        """Check if the update is aborted

        As long as the template file cannot be loaded by configparse, abort
        immediately.

        From user's perspective, it is similar with aborting commit when option
        -m is omitted. If all lines are commented out, abort.
        """
        config = configparser.ConfigParser()
        try:
            loaded_files = config.read(template_file)
        except configparser.MissingSectionHeaderError:
            return True
        # Something wrong with the template, which causes it cannot be loaded.
        if not loaded_files:
            return True
        # template can be loaded even if it's empty.
        if not config.sections():
            return True
        return False

    @staticmethod
    def _extract_issue_ids(text):
        """
        matching based on slightly modified expression taken from there:
        https://github.com/fedora-infra/bodhi/blob/5.3/bodhi/server/config.py

        Examples:
        Fixes: rhbz#11111
        Fix:  rh#22222
        resolves:rhbz#33333, rhbz#44444 rh#55555
        close: fedora#6666
        fix:epel#77777
        """
        # matches one Bugzilla bug ID (in case there are more IDs with one common prefix)
        id_pattern_raw = r'(?:fedora|epel|rh(?:bz)?)#(\d{5,})'
        bz_pattern = re.compile(
            # says: there is at least one complete (including prefix) Bugzilla bug occurrence
            r'(?:fix(?:es)?|close(?:s)?|resolve(?:s)?)(?::|:\s+|\s+)' + id_pattern_raw,
            re.IGNORECASE,
        )
        id_pattern = re.compile(id_pattern_raw, re.IGNORECASE)

        issue_ids = set()
        for line in text.splitlines():
            # is there complete Bugzilla ID including prefix
            bz_match = bz_pattern.search(line)
            if bz_match is not None:
                # gather all Bugzilla IDs behind the prefix
                line_ids = re.findall(id_pattern, line)
                issue_ids.update(line_ids)

        return sorted(list(issue_ids))

    def _prepare_bodhi_template(self, template_file):
        try:
            nvr = self.cmd.nvr
        except rpkgError:
            # This is not an RPM, can't get NVR
            nvr = "FILL_IN_NVR_HERE"
        bodhi_args = {
            'nvr': nvr,
            'bugs': '',
            'display_name': '',
            'descr': 'Here is where you give an explanation of your update.',
            'request': self.args.request,
            'autokarma': str(self.args.autokarma),
            'stable_karma': self.args.stable_karma,
            'unstable_karma': self.args.unstable_karma,
            'close_bugs': str(self.args.close_bugs),
            'suggest': 'unspecified',
            'require_bugs': str(self.args.require_bugs),
            'require_testcases': str(self.args.require_testcases),
            'severity': self.args.severity,
        }

        if self.args.suggest_reboot:
            bodhi_args['suggest'] = 'reboot'
        elif self.args.suggest_logout:
            bodhi_args['suggest'] = 'logout'

        if self.args.update_type:
            bodhi_args['type_'] = self.args.update_type
        else:
            bodhi_args['type_'] = ''

        if self.cmd._isrpmautospec():
            clog = self.cmd._getcommitmessage()
        else:
            try:
                self.cmd.clog()
                clog_file = os.path.join(self.cmd.path, 'clog')
                with io.open(clog_file, encoding='utf-8') as f:
                    clog = f.read()
                os.unlink(clog_file)
            except rpkgError:
                # Not an RPM, no changelog to work with
                clog = ""

        if self.args.bugs:
            bodhi_args['bugs'] = ','.join(self.args.bugs)
        else:
            # Extract bug numbers from the latest changelog entry
            bugs = self._extract_issue_ids(clog)
            if bugs:
                bodhi_args['bugs'] = ','.join(bugs)

        if self.args.notes:
            bodhi_args['descr'] = self.args.notes.replace('\n', '\n    ')
            bodhi_args['changelog'] = ''
        else:
            # Use clog as default message
            bodhi_args['descr'], bodhi_args['changelog'] = \
                self._format_update_clog(clog)

        template = textwrap.dedent(BODHI_TEMPLATE) % bodhi_args

        with io.open(template_file, 'w', encoding='utf-8') as f:
            f.write(template)

        if not self.args.update_type or not self.args.notes:
            # Open the template in a text editor
            editor = os.getenv('EDITOR', 'vi')
            cmd = editor.split()  # EDITOR command can contain additional arguments like "emacs -nw"
            cmd.append(template_file)
            self.cmd._run_command(cmd, shell=False)

            # Check to see if we got a template written out.  Bail otherwise
            if not os.path.isfile(template_file):
                raise rpkgError('No bodhi update details saved!')

            return not self.is_update_aborted(template_file)

        return True

    def update(self):
        check_bodhi_version()
        bodhi_config = self._get_bodhi_config()

        bodhi_template_file = 'bodhi.template'
        ready = self._prepare_bodhi_template(bodhi_template_file)

        if ready:
            try:
                self.cmd.update(bodhi_config, template=bodhi_template_file)
            except Exception as e:
                # Reserve original edited bodhi template so that packager could
                # have a chance to recover content on error for next try.
                shutil.copyfile(bodhi_template_file,
                                '{0}.last'.format(bodhi_template_file))
                raise rpkgError('Could not generate update request: %s\n'
                                'A copy of the filled in template is saved '
                                'as bodhi.template.last' % e)
            finally:
                os.unlink(bodhi_template_file)
        else:
            self.log.info('Bodhi update aborted!')

    def request_repo(self):
        self._request_repo(
            logger=self.log,
            repo_name=self.args.name,
            ns=self.args.new_repo_namespace,
            branch='rawhide',
            summary=self.args.summary,
            description=self.args.description,
            upstreamurl=self.args.upstreamurl,
            monitor=self.args.monitor,
            bug=self.args.bug,
            exception=self.args.exception,
            name=self.name,
            config=self.config,
            project_name=self.args.project_name,
            backend=self.args.backend,
            initial_commit=not self.args.no_initial_commit,
            onboard_packit=self.args.onboard_packit,
        )

    def request_tests_repo(self):
        self._request_repo(
            logger=self.log,
            repo_name=self.args.name,
            ns='tests',
            description=self.args.description,
            bug=self.args.bug,
            name=self.name,
            config=self.config,
            anongiturl=self.cmd.anongiturl,
        )

    @staticmethod
    def _request_repo(logger, repo_name, ns, description, name, config,
                      branch=None, summary=None, upstreamurl=None,
                      monitor=None, bug=None, exception=None, anongiturl=None,
                      project_name=None, backend=None,
                      initial_commit=True, onboard_packit=None,):
        """ Implementation of `request_repo`.

        Submits a request for a new dist-git repo.

        :param logger: A logger object.
        :param repo_name: The repository name string.  Typically the
            value of `self.cmd.repo_name`.
        :param ns: The repository namespace string, i.e. 'rpms' or 'modules'.
            Typically takes the value of `self.cmd.ns`.
        :param description: A string, the description of the new repo.
            Typically takes the value of `self.args.description`.
        :param name: A string representing which section of the config should be
            used.  Typically the value of `self.name`.
        :param config: A dict containing the configuration, loaded from file.
            Typically the value of `self.config`.
        :param branch: The git branch string when requesting a repo.
            Typically 'rawhide'.
        :param summary: A string, the summary of the new repo.  Typically
            takes the value of `self.args.summary`.
        :param upstreamurl: A string, the upstreamurl of the new repo.
            Typically takes the value of `self.args.upstreamurl`.
        :param monitor: A string, the monitoring flag of the new repo, i.e.
            `'no-monitoring'`, `'monitoring'`, or `'monitoring-with-scratch'`.
            Typically takes the value of `self.args.monitor`.
        :param bug: An integer representing the bugzilla ID of a "package
            review" associated with this new repo.  Typically takes the
            value of `self.args.bug`.
        :param exception: An boolean specifying whether or not this request is
            an exception to the packaging policy.  Exceptional requests may be
            granted the right to waive their package review at the discretion of
            Release Engineering.  Typically takes the value of
            `self.args.exception`.
        :param project_name: A string representing project name for Anitya
            if It's different from package name in distribution
        :param backend: A string representing the backend creating Anitya project:
        :param anongiturl: A string with the name of the anonymous git url.
            Typically the value of `self.cmd.anongiturl`.
        :param onboard_packit: A string, whether to generate a default
            Packit configuration for the package and open a pull request
            against or push directly to the newly created dist-git repository.
            Accepted values are `no`, `pull-request` and `push`.
            Typically takes the value of `self.args.onboard_packit`.
        :return: None
        """
        # bug is not a required parameter in the event the packager has an
        # exception, in which case, they may use the --exception flag
        # neither in case of modules, which don't require a formal review
        if not bug and not exception and ns not in ['tests', 'modules', 'flatpaks']:
            raise rpkgError(
                'A Bugzilla bug is required on new repository requests')
        repo_regex = r'^[a-zA-Z0-9_][a-zA-Z0-9-_.+]*$'
        if not bool(re.match(repo_regex, repo_name)):
            raise rpkgError(
                'The repository name "{0}" is invalid. It must be at least '
                'two characters long with only letters, numbers, hyphens, '
                'underscores, plus signs, and/or periods. Please note that '
                'the project cannot start with a period or a plus sign.'
                .format(repo_name))

        summary_from_bug = ''
        if bug and ns not in ['modules', 'flatpaks']:
            bz_url = config.get('{0}.bugzilla'.format(name), 'url')
            bz_client = BugzillaClient(bz_url)
            bug_obj = bz_client.get_review_bug(bug, ns, repo_name)
            summary_from_bug = bug_obj.summary.split(' - ', 1)[1].strip()

        if ns == 'tests':
            # check if tests repository does not exist already
            assert_new_tests_repo(repo_name, get_dist_git_url(anongiturl))

            ticket_body = {
                'action': 'new_repo',
                'branch': 'main',
                'bug_id': bug or '',
                'monitor': 'no-monitoring',
                'namespace': 'tests',
                'onboard_packit': 'no',
                'repo': repo_name,
                'description': description,
                'upstreamurl': '',
            }
        else:
            # Default branch is rawhide, but for flatpaks namespace its 'stable'
            if ns == 'flatpaks':
                branch = 'stable'
            ticket_body = {
                'action': 'new_repo',
                'branch': branch,
                'bug_id': bug or '',
                'description': description or '',
                'exception': exception,
                'monitor': monitor,
                'namespace': ns,
                'onboard_packit': onboard_packit,
                'repo': repo_name,
                'summary': summary or summary_from_bug,
                'upstreamurl': upstreamurl or ''
            }
            if not initial_commit:
                ticket_body['initial_commit'] = False

        if monitor != "no-monitoring":
            if not does_package_exist_in_anitya(repo_name):
                if backend is None or upstreamurl == '' or upstreamurl is None:
                    raise rpkgError(
                        "When monitoring or monitoring-with-scratch arguments are set "
                        "you are also required to provide `--backend`, `--upstreamurl` "
                        "and `--project-name` (if it's different from package name) arguments "
                        "to create a project and package for Anitya."
                    )
            if project_name is None:
                project_name = repo_name
            if backend is None:
                backend = "custom"
            ticket_body.update({
                'backend': backend,
                'project_name': project_name,
                'distribution': 'Fedora',
            })

        ticket_body = json.dumps(ticket_body, indent=True)
        ticket_body = '```\n{0}\n```'.format(ticket_body)
        ticket_title = 'New Repo for "{0}/{1}"'.format(ns, repo_name)

        pagure_section = '{0}.pagure'.format(name)
        pagure_url = config_get_safely(config, pagure_section, 'url')
        pagure_token = config_get_safely(config, pagure_section, 'token')
        print(new_pagure_issue(
            logger, pagure_url, pagure_token, ticket_title, ticket_body, name))

    def request_branch(self):
        if self.args.repo_name_for_branch:
            self.cmd.repo_name = self.args.repo_name_for_branch
            self.cmd.ns = self.args.repo_ns_for_branch or 'rpms'

        try:
            active_branch = self.cmd.repo.active_branch.name
        except rpkgError:
            active_branch = None
        self._request_branch(
            logger=self.log,
            service_levels=self.args.sl,
            all_releases=self.args.all_releases,
            branch=self.args.branch,
            active_branch=active_branch,
            repo_name=self.cmd.repo_name,
            ns=self.cmd.ns,
            no_git_branch=self.args.no_git_branch,
            no_auto_module=self.args.no_auto_module,
            name=self.name,
            config=self.config,
            anongiturl=self.cmd.anongiturl,
        )

    @staticmethod
    def _request_branch(logger, service_levels, all_releases, branch,
                        active_branch, repo_name, ns, no_git_branch,
                        no_auto_module, name, config, anongiturl):
        """ Implementation of `request_branch`.

        Submits a request for a new branch of a given dist-git repo.

        :param logger: A logger object.
        :param service_levels: A list of service level strings.  Typically the
            value of `self.args.service_levels`.
        :param all_releases: A boolean indicating if this request should be made
            for all active Fedora branches.
        :param branch: A string specifying the specific branch to be requested.
        :param active_branch: A string (or None) specifying the active branch in
            the current git repo (the branch that is currently checked out).
        :param repo_name: The repository name string.  Typically the
            value of `self.cmd.repo_name`.
        :param ns: The repository namespace string, i.e. 'rpms' or 'modules'.
            Typically takes the value of `self.cmd.ns`.
        :param no_git_branch: A boolean flag.  If True, the SCM admins should
            create the git branch in PDC, but not in pagure.io.
        :param no_auto_module: A boolean flag.  If True, requests for
            non-standard branches should not automatically result in additional
            requests for matching modules.
        :param name: A string representing which section of the config should be
            used.  Typically the value of `self.name`.
        :param config: A dict containing the configuration, loaded from file.
            Typically the value of `self.config`.
        :param anongiturl: A string with the name of the anonymous git url.
            Typically the value of `self.cmd.anongiturl`.
        :return: None
        """

        def get_branch():
            """Returns the branch according to inputs."""
            if next_match:
                return next_match.groups()[0]
            return branch

        def check_input_parameters():
            """Checks input parameters.

            If there is not a valid combination of input parameter values
            the error message will be set.
            """
            if all_releases and branch:
                return 'You cannot specify a branch with the ' \
                       '"--all-releases" option'
            if all_releases and not branch and service_levels:
                return 'You cannot specify service levels with the ' \
                       '"--all-releases" option'
            if not all_releases and not branch and not active_branch:
                return 'You must specify a branch if you are not in ' \
                       'a git repository'

        def check_branch(branch):
            if branch in release_branches:
                if service_levels:
                    return 'You can\'t provide SLs for release branches'
            else:
                if re.match(RELEASE_BRANCH_REGEX, branch):
                    return '{0} is release branch, but not active'.format(branch)
                elif not service_levels:
                    return 'You must provide SLs for non-release branches {0}'.format(branch)

        status_message = check_input_parameters()
        if status_message:
            raise rpkgError(status_message)

        if not (branch or all_releases) and active_branch:
            branch = active_branch

        bodhi_url = config.get('{0}.bodhi'.format(name), 'url')
        if branch:
            if is_epel(branch):
                assert_valid_epel_package(repo_name, branch)

            if ns in ['modules', 'test-modules', 'flatpaks']:
                branch_valid = bool(re.match(r'^[a-zA-Z0-9.\-_+]+$', branch))
                if not branch_valid:
                    raise rpkgError(
                        'Only characters, numbers, periods, dashes, '
                        'underscores, and pluses are allowed in {0} branch '
                        'names'.format('flatpak' if ns == 'flatpaks' else 'module'))
            release_branches = list(itertools.chain(
                *list(get_release_branches(bodhi_url).values())))
            # treat epel*-next the same as epel* release branches
            next_match = re.match(r'^(epel\d+)-next$', branch)

            branch_error_message = check_branch(get_branch())
            if branch_error_message:
                raise rpkgError(branch_error_message)

        # If service levels were provided, verify them
        if service_levels:
            sl_dict = sl_list_to_dict(service_levels)
            verify_sls(sl_dict)

        pagure_section = '{0}.pagure'.format(name)
        pagure_url = config_get_safely(config, pagure_section, 'url')
        pagure_token = config_get_safely(config, pagure_section, 'token')
        if all_releases:
            release_branches = list(itertools.chain(
                *list(get_release_branches(bodhi_url).values())))
            branches = [b for b in release_branches
                        if re.match(r'^(f\d+)$', b)]
        else:
            branches = [branch]

        for b in sorted(list(branches), reverse=True):
            ticket_body = {
                'action': 'new_branch',
                'branch': b,
                'namespace': ns,
                'repo': repo_name,
                'create_git_branch': not no_git_branch
            }
            if service_levels:
                ticket_body['sls'] = sl_dict

            ticket_body = json.dumps(ticket_body, indent=True)
            ticket_body = '```\n{0}\n```'.format(ticket_body)
            ticket_title = 'New Branch "{0}" for "{1}/{2}"'.format(
                b, ns, repo_name)

            # check whether the requested branch was already created
            if b not in get_pagure_branches(
                logger=logger,
                url=get_dist_git_url(anongiturl),
                namespace=ns,
                repo_name=repo_name
            ):
                print(new_pagure_issue(
                    logger, pagure_url, pagure_token, ticket_title, ticket_body,
                    name))
            else:
                logger.warning('Requested branch "{0}" already exists. '
                               'Skipping the operation.'.format(b))

            # For non-standard rpm branch requests, also request a matching new
            # module repo with a matching branch.
            auto_module = (
                ns == 'rpms'
                and not re.match(RELEASE_BRANCH_REGEX, b)
                and not next_match        # Dont run auto_module on epel-next requests
                and not no_auto_module
            )
            if auto_module:
                summary = ('Automatically requested module for '
                           'rpms/%s:%s.' % (repo_name, b))
                fedpkgClient._request_repo(
                    logger=logger,
                    repo_name=repo_name,
                    ns='modules',
                    branch='rawhide',
                    summary=summary,
                    description=summary,
                    upstreamurl=None,
                    monitor='no-monitoring',
                    bug=None,
                    exception=True,
                    name=name,
                    config=config,
                    onboard_packit='no',
                )
                fedpkgClient._request_branch(
                    logger=logger,
                    service_levels=service_levels,
                    all_releases=all_releases,
                    branch=b,
                    active_branch=active_branch,
                    repo_name=repo_name,
                    ns='modules',
                    no_git_branch=no_git_branch,
                    no_auto_module=True,  # Avoid infinite recursion.
                    name=name,
                    config=config,
                    anongiturl=anongiturl,
                )

    def request_unretirement(self):
        if self.args.repo_name:
            self.cmd.repo_name = self.args.repo_name
            self.cmd.ns = self.args.repo_ns_name

        try:
            active_branch = self.cmd.repo.active_branch.name
        except rpkgError:
            active_branch = None

        if not self.args.branches:
            if not active_branch:
                branches = ["rawhide"]
            else:
                branches = [active_branch]
        else:
            branches = self.args.branches

        self._request_unretirement(
            logger=self.log,
            repo_name=self.cmd.repo_name,
            ns=self.cmd.ns,
            branches=branches,
            bugzila_url=self.args.bz_bug_id,
            fas_name=self.cmd.user,
            name=self.name,
            config=self.config,
        )

    @staticmethod
    def _request_unretirement(
            logger, repo_name, ns, branches, bugzila_url, fas_name, name, config
    ):
        """ Implementation of `request_unretirement`.

        Submits a request for the unretirement of the package branch.

        :param logger: A logger object.
        :param repo_name: The string of the repo name.
        :param ns: The string of pacakge namespace.
        :param branches: The list of branches that need to be unretired.
        :param bugzila_url: The URL of the bugzilla review.
            Typically, the value of `self.args.bz_url`, None if not needed.
        :param fas_name: The string of fas name of user.
            Typically, value is `self.cmd.user`.
        :param name: A string representing which section of the config should be
            used. Typically, the value of `self.name`.
        :param config: A dict containing the configuration, loaded from file.
            Typically, the value of `self.config`.
        """
        distgit_section = '{0}.distgit'.format(name)
        distgit_url = config_get_safely(config, distgit_section, 'apibaseurl')
        bodhi_url = config.get('{0}.bodhi'.format(name), 'url')
        current_time = datetime.now()
        bugzilla_need = False

        release_branches = list(itertools.chain(
            *list(get_release_branches(bodhi_url).values())))
        release_branches.append('rawhide')

        requested_branches = branches.copy()
        for branch in requested_branches:
            if branch not in release_branches:
                branches.remove(branch)
                print(
                    f"Branch {0} can't be unretired, because it is not a release branch."
                    .format(branch)
                )

        if not branches:
            raise rpkgError(
                "No requested branches {0} are release branches."
                " So there is nothing to unretire.".format(requested_branches)
            )

        for branch in branches:
            commit_date = get_last_commit_date(distgit_url, ns, repo_name, branch)
            committed_time = datetime.fromtimestamp(commit_date)
            difference_in_days = (current_time - committed_time).days
            if difference_in_days > 56:  # 56 days is 8 weeks
                bugzilla_need = True
                if bugzila_url is None:
                    raise rpkgError(
                        "Bugzilla url should be provided, "
                        "because last commit was made more than 8 weeks ago."
                    )

        # is bug has fedora-review+ flag
        def get_bug_id(bz_info):
            if not bz_info:
                return None
            match = re.search(BUGZILLA_URL_REGEX, bz_info)
            if match:
                bz_bug_id = match.group(1)
                return bz_bug_id
            else:
                return bz_info

        bug_id = get_bug_id(bugzila_url)

        if bugzilla_need:
            bz_url = config.get('{0}.bugzilla'.format(name), 'url')
            bz_client = BugzillaClient(bz_url)
            # get_review_bug checks if the bug has fedora_review+ flag
            bz_client.get_review_bug(bug_id, ns, repo_name)

        # Ticket creation
        pagure_section = '{0}.pagure'.format(name)
        pagure_url = config_get_safely(config, pagure_section, 'url')
        pagure_token = config_get_safely(config, pagure_section, 'token')

        ticket_body = {
            'action': 'unretirement',
            'name': repo_name,
            'type': ns,
            'branches': branches,
            'review_bugzilla': bug_id,
            'maintainer': fas_name,
        }
        ticket_body = json.dumps(ticket_body, indent=True)
        ticket_body = '```\n{0}\n```'.format(ticket_body)
        ticket_title = 'Unretire {0}'.format(repo_name)

        print(
            new_pagure_issue(
                logger=logger,
                url=pagure_url,
                token=pagure_token,
                title=ticket_title,
                body=ticket_body,
                cli_name=name,
            )
        )

    def do_distgit_fork(self):
        """create fork of the distgit repository
        That includes creating fork itself using API call and then adding
        remote tracked repository
        """
        anon = self.args.anonymous
        if self.cmd.push_url.startswith('https://'):
            anon = True
            self.log.info('Detected anonymously cloned repo (`fedpkg clone -a`) '
                          '- forking in the anonymous mode.')
        distgit_section = '{0}.distgit'.format(self.name)
        distgit_api_base_url = config_get_safely(self.config, distgit_section, "apibaseurl")
        distgit_remote_base_url = self.config.get(
            '{0}'.format(self.name),
            'anongiturl' if anon else 'gitbaseurl',
            vars={'user': self.cmd.user, 'repo': self.cmd.repo_name},
        )
        distgit_token = config_get_safely(self.config, distgit_section, 'token')

        ret = do_fork(
            logger=self.log,
            base_url=distgit_api_base_url,
            token=distgit_token,
            repo_name=self.cmd.repo_name,
            namespace=self.cmd.ns,
            cli_name=self.name,
        )

        # assemble url of the repo in web browser
        fork_url = '{0}/fork/{1}/{2}/{3}'.format(
            distgit_api_base_url.rstrip('/'),
            self.cmd.user,
            self.cmd.ns,
            self.cmd.repo_name,
        )

        if ret:
            msg = "Fork of the repository has been created: '{0}'"
        else:
            msg = "Repo '{0}' already exists."
        self.log.info(msg.format(fork_url))

        ret = do_add_remote(
            base_url=distgit_api_base_url,
            remote_base_url=distgit_remote_base_url,
            username=self.cmd.user,
            repo=self.cmd.repo,
            repo_name=self.cmd.repo_name,
            namespace=self.cmd.ns,
        )
        if ret:
            msg = "Adding as remote '{0}'."
        else:
            msg = "Remote with name '{0}' already exists."
        self.log.info(msg.format(self.cmd.user))

    def create_buildroot_override(self):
        """Create a buildroot override in Bodhi"""
        check_bodhi_version()
        if self.args.NVR:
            if not self.cmd.anon_kojisession.getBuild(self.args.NVR):
                raise rpkgError(
                    'Build {0} does not exist.'.format(self.args.NVR))
        bodhi_config = self._get_bodhi_config()
        self.cmd.create_buildroot_override(
            bodhi_config,
            build=self.args.NVR or self.cmd.nvr,
            duration=self.args.duration,
            notes=self.args.notes)

    def extend_buildroot_override(self):
        check_bodhi_version()
        if self.args.NVR:
            if not self.cmd.anon_kojisession.getBuild(self.args.NVR):
                raise rpkgError(
                    'Build {0} does not exist.'.format(self.args.NVR))
        bodhi_config = self._get_bodhi_config()
        self.cmd.extend_buildroot_override(
            bodhi_config,
            build=self.args.NVR or self.cmd.nvr,
            duration=self.args.duration)

    def read_releases_from_local_config(self, active_releases):
        """Read configured releases from build config from repo"""
        config_file = os.path.join(self.cmd.path, LOCAL_PACKAGE_CONFIG)
        if not os.path.exists(config_file):
            return None
        config = configparser.ConfigParser()
        if not config.read([config_file]):
            raise rpkgError('Package config {0} is not accessible.'.format(
                LOCAL_PACKAGE_CONFIG))
        if not config.has_option('koji', 'targets'):
            self.log.warning(
                'Build target is not configured. Continue to build as normal.')
            return None
        target_releases = config.get('koji', 'targets', raw=True).split()
        expanded_releases = []
        for rel in target_releases:
            expanded = expand_release(rel, active_releases)
            if expanded:
                expanded_releases += expanded
            else:
                self.log.error('Target %s is unknown. Skip.', rel)
        return sorted(set(expanded_releases))

    @staticmethod
    def is_stream_branch(stream_branches, name):
        """Determine if a branch is stream branch

        :param stream_branches: list of stream branches of a package. Each of
            them is a mapping containing name and active status, which are
            minimum set of properties to be included. For example, ``[{'name':
            '8', 'active': true}, {'name': '10', 'active': true}]``.
        :type stream_branches: list[dict]
        :param str name: branch name to check if it is a stream branch.
        :return: True if branch is a stream branch, False otherwise.
        :raises rpkgError: if branch is a stream branch but it is inactive.
        """
        for branch_info in stream_branches:
            if branch_info == name:
                return True
        return False

    def _build(self, sets=None):
        if hasattr(self.args, 'chain') or self.args.scratch:
            return super(fedpkgClient, self)._build(sets)

        server_url = self.config.get('{0}.bodhi'.format(self.name), 'url')
        distgit_section = '{0}.distgit'.format(self.name)
        apibaseurl = config_get_safely(self.config, distgit_section, "apibaseurl")
        logger = self.log

        stream_branches = get_stream_branches(server_url, self.cmd.repo_name, apibaseurl, logger)
        self.log.debug(
            'Package %s has stream branches: %r',
            self.cmd.repo_name, [item for item in stream_branches])

        # rawhide was removed from the list of stream branches, but for rawhide branch
        # the local config should be checked and therefore it has to skip this condition
        if not self.is_stream_branch(stream_branches + ['rawhide'], self.cmd.branch_merge):
            return super(fedpkgClient, self)._build(sets)

        self.log.debug('Current branch %s is a stream branch.',
                       self.cmd.branch_merge)

        releases = self.read_releases_from_local_config(
            get_release_branches(server_url))

        if not releases:
            # If local config file is not created yet, or no build targets
            # are not configured, let's build as normal.
            return super(fedpkgClient, self)._build(sets)

        self.log.debug('Build on release targets: %r', releases)
        task_ids = []
        for release in releases:
            self.cmd.branch_merge = release
            self.cmd.target = self.cmd.build_target(release)
            # self.rel has to be regenerated by self.load_nameverrel, because it differs
            # for every release. It is used in nvr-already-built check (self.nvr) later.
            self.cmd.load_nameverrel()
            task_id = super(fedpkgClient, self)._build(sets)
            task_ids.append(task_id)
        return task_ids

    def show_releases_info(self):
        server_url = self.config.get('{0}.bodhi'.format(self.name), 'url')
        releases = get_release_branches(server_url)

        def _join(ln):
            return ' '.join(ln)

        if self.args.show_epel_only:
            print(_join(releases['epel']))
        elif self.args.show_fedora_only:
            print(_join(releases['fedora']))
        elif self.args.join:
            print(' '.join(itertools.chain(releases['fedora'],
                                           releases['epel'])))
        else:
            print('Fedora: {0}'.format(_join(releases['fedora'])))
            print('EPEL: {0}'.format(_join(releases['epel'])))

    def _check_token(self, token, token_type):

        if token is None or token == "":
            self.log.error("ERROR: No input.")
            return False

        match = re.search(r'^\s*[A-Z0-9]{64}\s*$', token)
        if match is None:
            self.log.error("ERROR: Token is not properly formatted.")
            return False
        else:
            return True

    def set_pagure_token(self):
        super(fedpkgClient, self)._set_token("pagure")

    def set_distgit_token(self):
        super(fedpkgClient, self)._set_token("distgit")

    def retire(self):
        """
        Runs the rpkg retire command after check. Check includes reading the state
        of Fedora release.
        """
        # Allow retiring in epel
        if is_epel(self.cmd.branch_merge):
            super(fedpkgClient, self).retire()
        else:
            state = get_fedora_release_state(self.config, self.name, self.cmd.branch_merge)

            # Allow retiring in Rawhide and Branched until Final Freeze
            if state is None or state == 'pending':
                super(fedpkgClient, self).retire()
            else:
                self.log.error("Fedora release (%s) is in state '%s' - retire operation "
                               "is not allowed." % (self.cmd.branch_merge, state))

        if self.cmd.is_retired():
            self.log.info("To disable monitoring of the (retired) repository use manually "
                          "the separate command: '{0} disable-monitoring'".format(self.name))

    def do_disable_monitoring(self):
        """
        Disable monitoring when repository is retired.
        """
        distgit_section = '{0}.distgit'.format(self.name)
        distgit_api_base_url = config_get_safely(self.config, distgit_section, "apibaseurl")
        distgit_token = config_get_safely(self.config, distgit_section, 'token')
        try:
            if self.cmd.is_retired():
                disable_monitoring(
                    logger=self.log,
                    base_url=distgit_api_base_url,
                    token=distgit_token,
                    repo_name=self.cmd.repo_name,
                    namespace=self.cmd.ns,
                    cli_name=self.name,
                )
            else:
                self.log.error("ERROR: the repository is not yet retired.")
                return 1
        except Exception:
            self.log.error(
                "In case of failure, please look at the help command "
                "'{0} disable-monitoring --help'.\n".format(self.name))
            raise
