# -*- coding: utf-8 -*-
# fedpkg - a Python library for RPM Packagers
#
# Copyright (C) 2017 Red Hat Inc.
# Author(s): Chenxiong Qi <cqi@redhat.com>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
# the full text of the license.

import argparse
import io
import json
import os
import re
import sys
import unittest
from configparser import NoOptionError, NoSectionError
from datetime import datetime, timedelta, timezone
from os import rmdir
from tempfile import mkdtemp, mkstemp
from unittest.mock import Mock, PropertyMock, call, patch

import git

# Use deprecated pkg_resources if packaging library isn't available (python 3.6)
try:
    from packaging.version import parse as parse_version
except ImportError:
    from pkg_resources import parse_version
# Use deprecated pkg_resources if importlib isn't available (python 3.6)
try:
    from importlib.metadata import distribution
except ImportError:
    from pkg_resources import get_distribution as distribution

import fedpkg.cli
from fedpkg.bugzilla import BugzillaClient
from fedpkg.cli import check_bodhi_version
from freezegun import freeze_time
from pyrpkg.errors import rpkgError
from utils import CliTestCase

try:
    import bodhi
    bodhi_version = distribution('bodhi-client').version
    if parse_version(bodhi_version) < parse_version("6.0.0"):
        raise ImportError("Unsupported bodhi-client")
except ImportError:
    bodhi = None


def _mock_metadata(self, _):
    """Minimal replacement for OIDCClient._get_provider_metadata"""
    self.metadata = {"token_endpoint": "", "authorization_endpoint": ""}


class TestIsUpdateAborted(CliTestCase):
    """Test is_update_aborted"""

    require_test_repos = False

    def setUp(self):
        fd, self.bodhi_template = mkstemp()
        os.close(fd)

    def tearDown(self):
        os.unlink(self.bodhi_template)

    def _is_update_aborted(self):
        with patch('sys.argv', new=['fedpkg', 'update']):
            cli = self.new_cli()
        return cli.is_update_aborted(self.bodhi_template)

    def test_template_is_emtpy(self):
        self.assertTrue(self._is_update_aborted())

    def test_all_line_are_commented_out(self):
        with io.open(self.bodhi_template, 'w', encoding='utf-8') as f:
            f.write('# line 1\n#line 2\n#line 3\n')

        self.assertTrue(self._is_update_aborted())

    def test_template_is_ok(self):
        with io.open(self.bodhi_template, 'w', encoding='utf-8') as f:
            f.write('[fedpkg-1.34-1.fc28]\ntype=\nnotes=abc\n')

        self.assertFalse(self._is_update_aborted())

    def test_template_content_is_broken(self):
        with io.open(self.bodhi_template, 'w', encoding='utf-8') as f:
            f.write('#[fedpkg-1.34-1.fc28]\ntype=\nnotes=abc\n')

        self.assertTrue(self._is_update_aborted())


@unittest.skipUnless(bodhi, 'Skip if no supported bodhi-client is available')
class TestUpdate(CliTestCase):
    """Test update command"""

    create_repo_per_test = False

    def setUp(self):
        super(TestUpdate, self).setUp()

        self.nvr_patcher = patch('fedpkg.Commands.nvr',
                                 new_callable=PropertyMock,
                                 return_value='fedpkg-1.29-9.fc27')
        self.mock_nvr = self.nvr_patcher.start()

        self.run_command_patcher = patch('fedpkg.Commands._run_command')
        self.mock_run_command = self.run_command_patcher.start()

        # Let's always use the bodhi 2 command line to test here
        self.check_bodhi_version_patcher = patch(
            'fedpkg.cli.check_bodhi_version')
        self.mock_check_bodhi_version = \
            self.check_bodhi_version_patcher.start()

        self.tempdir = mkdtemp()
        self.os_environ_patcher = patch.dict('os.environ', {'EDITOR': 'vi', 'HOME': self.tempdir})
        self.os_environ_patcher.start()

        self.user_patcher = patch('pyrpkg.Commands.user',
                                  return_value='someone')
        self.mock_user = self.user_patcher.start()

        # Not write clog actually.
        self.clog_patcher = patch('fedpkg.Commands.clog')
        self.clog_patcher.start()

        # Logs will be read in the tests which do not specify --notes option
        self.fake_clog = [
            'Add tests for command update',
            'New command update - #1000',  # invalid bug id format
            'Fix tests - #2000, #notabug',  # both invalid bug id format
            '处理一些Unicode字符číář',
            'fix: rh#10001',
            'Fixes: rhbz#20001',
            ' close: fedora#30001',  # test whitespace at the beginning
        ]
        clog_file = os.path.join(self.cloned_repo_path, 'clog')
        with io.open(clog_file, 'w', encoding='utf-8') as f:
            f.write(os.linesep.join(self.fake_clog))

        self.oidcmeta_patcher = patch(
            'bodhi.client.oidcclient.OIDCClient._get_provider_metadata', _mock_metadata
        )
        self.oidcmeta_patcher.start()

    def tearDown(self):
        if os.path.exists('bodhi.template'):
            os.unlink('bodhi.template')
        if os.path.exists('bodhi.template.last'):
            os.unlink('bodhi.template.last')
        os.unlink(os.path.join(self.cloned_repo_path, 'clog'))
        self.user_patcher.stop()
        self.os_environ_patcher.stop()
        self.clog_patcher.stop()
        self.check_bodhi_version_patcher.stop()
        self.run_command_patcher.stop()
        self.nvr_patcher.stop()
        self.oidcmeta_patcher.stop()
        rmdir(self.tempdir)
        super(TestUpdate, self).tearDown()

    def get_cli(self, cli_cmd, name='fedpkg', cfg=None):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg)

    @patch('bodhi.client.bindings.BodhiClient.csrf')
    @patch('bodhi.client.bindings.BodhiClient.send_request')
    def assert_bodhi_update(self, cli, send_request, csrf,
                            update_type=None, request_type=None, notes=None,
                            stable_karma=None, unstable_karma=None,
                            suggest=None, severity=None):
        csrf.return_value = '123456'

        def run_command_side_effect(command, shell):
            # The real call accepts first argument as ['vi', 'bodhi.template'],
            # command[-1] returns the filename.
            filename = command[-1]
            with io.open(filename, 'r', encoding='utf-8') as f:
                content = f.read()
            with io.open(filename, 'w', encoding='utf-8') as f:
                # Update parameters here for test
                if update_type:
                    content = content.replace(
                        'type=\n', 'type={0}\n'.format(update_type))
                if request_type:
                    # CLI option --request has default value, so the template
                    # contains the default value.
                    content = re.sub('request=[a-z]+\n',
                                     'request={0}\n'.format(request_type),
                                     content)
                if suggest:
                    content = re.sub('suggest=[a-z]+\n',
                                     'suggest={0}\n'.format(suggest),
                                     content)
                f.write(content)

        self.mock_run_command.side_effect = run_command_side_effect

        expected_data = {
            'autokarma': 'True',
            'bugs': '10001,20001,30001',
            'display_name': '',
            'builds': ' {0} '.format(self.mock_nvr.return_value),
            'close_bugs': True,
            'request': 'testing',
            'severity': severity or 'unspecified',
            'suggest': 'unspecified',
            'type': update_type,
            'type_': update_type,
            'stable_karma': stable_karma or '3',
            'unstable_karma': unstable_karma or '-3',
            'csrf_token': csrf.return_value,
        }
        if notes:
            expected_data['notes'] = notes
        else:
            expected_data['notes'] = self.fake_clog[0]

        # there wasn't the option in older releases of bodhi, but these
        # releases are still active (epel7, epel8)
        if parse_version(bodhi_version) <= parse_version("4.0.0"):
            del expected_data["display_name"]

        with patch('os.unlink') as unlink:
            cli.update()

            csrf.assert_called_once_with()
            send_request.assert_called_once_with(
                'updates/', verb='POST', auth=True, data=expected_data)

            unlink.assert_has_calls([
                call('bodhi.template'),
                call(os.path.join(cli.cmd.path, 'clog'))
            ], any_order=True)

        with io.open('bodhi.template', encoding='utf-8') as f:
            bodhi_template = f.read()
        self.assertTrue(self.mock_nvr.return_value in bodhi_template)
        self.assertTrue('10001,20001,30001' in bodhi_template)
        if notes:
            self.assertTrue(notes.replace('\n', '\n    ') in bodhi_template)
        else:
            self.assertTrue(self.fake_clog[0] in bodhi_template)
            rest_clog = os.linesep.join([
                '# {0}'.format(line) for line in self.fake_clog[1:]
            ])
            self.assertTrue(rest_clog in bodhi_template)

    def test_fail_if_missing_config_options(self):
        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)

        with patch.object(cli.config, 'get',
                          side_effect=NoOptionError('url', 'bodhi')):
            self.assertRaisesRegex(
                rpkgError, 'Could not get bodhi options.', cli.update)

        with patch.object(cli.config, 'get',
                          side_effect=NoSectionError('bodhi')):
            self.assertRaisesRegex(
                rpkgError, 'Could not get bodhi options.', cli.update)

    @patch('os.path.isfile', return_value=False)
    def test_fail_if_bodhi_template_is_not_a_file(self, isfile):
        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'update']

        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(
            rpkgError, 'No bodhi update details saved',
            self.assert_bodhi_update, cli)

        self.mock_run_command.assert_called_once_with(
            ['vi', 'bodhi.template'], shell=False)

    def test_request_update(self):
        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'update']

        cli = self.get_cli(cli_cmd)
        self.assert_bodhi_update(cli, update_type='enhancement')

        self.mock_run_command.assert_called_once_with(
            ['vi', 'bodhi.template'], shell=False)

    def test_request_update_with_karma(self):
        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path, 'update',
            '--stable-karma', '1', '--unstable-karma', '-1'
        ]

        cli = self.get_cli(cli_cmd)
        self.assert_bodhi_update(cli, update_type='enhancement',
                                 stable_karma='1', unstable_karma='-1')

    def test_request_security_update_with_severity(self):
        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path, 'update',
            '--type', 'security', '--severity', 'medium'
        ]

        cli = self.get_cli(cli_cmd)
        self.assert_bodhi_update(cli, update_type='security',
                                 severity='medium')

    @patch('fedpkg.Commands.update', side_effect=OSError)
    def test_handle_any_errors_raised_when_execute_bodhi(self, update):
        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'update']

        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(
            rpkgError, 'Could not generate update request',
            self.assert_bodhi_update, cli)

    def test_create_update_in_stage_bodhi(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd,
                           name='fedpkg-stage',
                           cfg='fedpkg-stage.conf')
        self.assert_bodhi_update(cli, update_type='enhancement')

        self.mock_run_command.assert_called_once_with(
            ['vi', 'bodhi.template'], shell=False)

    def test_missing_update_type_in_template(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(rpkgError, 'Missing update type',
                               self.assert_bodhi_update, cli)

    def test_incorrect_update_type_in_template(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(rpkgError, 'Incorrect update type',
                               self.assert_bodhi_update, cli, update_type='xxx')

    def test_incorrect_request_type_in_template(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(rpkgError, 'Incorrect request type',
                               self.assert_bodhi_update, cli,
                               update_type='enhancement',
                               request_type='xxx')

    def test_incorrect_suggest_type_in_template(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)
        self.assertRaisesRegex(rpkgError, 'Incorrect suggest type',
                               self.assert_bodhi_update, cli,
                               update_type='enhancement',
                               suggest='123')

    def test_create_with_cli_options(self):
        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path,
            'update', '--type', 'bugfix', '--notes', 'Mass rebuild'
        ]

        cli = self.get_cli(cli_cmd)

        self.assert_bodhi_update(
            cli, update_type='bugfix', notes='Mass rebuild')
        self.mock_run_command.assert_not_called()

    def test_show_editor_if_update_type_is_omitted(self):
        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path,
            'update', '--notes', 'Mass rebuild'
        ]

        cli = self.get_cli(cli_cmd)

        self.assert_bodhi_update(
            cli, update_type='enhancement', notes='Mass rebuild')
        self.mock_run_command.assert_called_once_with(
            ['vi', 'bodhi.template'], shell=False)

    def test_show_editor_if_notes_is_omitted(self):
        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path,
            'update', '--type', 'bugfix'
        ]

        cli = self.get_cli(cli_cmd)

        self.assert_bodhi_update(cli, update_type='bugfix')
        self.mock_run_command.assert_called_once_with(
            ['vi', 'bodhi.template'], shell=False)

    def test_create_with_notes_in_multiple_lines(self):
        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path,
            'update', '--type', 'bugfix', '--notes', 'Line 1\nLine 2\nLine 3'
        ]

        cli = self.get_cli(cli_cmd)

        self.assert_bodhi_update(
            cli, update_type='bugfix', notes='Line 1\nLine 2\nLine 3')

    @patch('sys.stderr', new=io.StringIO())
    def test_invalid_stable_karma_option(self):
        with self.assertRaises(SystemExit):
            self.get_cli([
                'fedpkg-stage', '--path', self.cloned_repo_path,
                'update', '--stable-karma', '10s'
            ])

        with self.assertRaises(SystemExit):
            self.get_cli([
                'fedpkg-stage', '--path', self.cloned_repo_path,
                'update', '--stable-karma', '-3'
            ])

    @patch('sys.stderr', new=io.StringIO())
    def test_invalid_unstable_karma_option(self):
        with self.assertRaises(SystemExit):
            self.get_cli([
                'fedpkg-stage', '--path', self.cloned_repo_path,
                'update', '--unstable-karma', '10s'
            ])

        with self.assertRaises(SystemExit):
            self.get_cli([
                'fedpkg-stage', '--path', self.cloned_repo_path,
                'update', '--unstable-karma', '3'
            ])

    @patch('sys.stderr', new=io.StringIO())
    def test_invalid_bug(self):
        with self.assertRaises(SystemExit):
            self.get_cli([
                'fedpkg-stage', '--path', self.cloned_repo_path,
                'update', '--bugs', '1000', '1001', '100l'
            ])

    def test_reserve_edited_template_on_error(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'update']
        cli = self.get_cli(cli_cmd)

        try:
            self.assert_bodhi_update(cli, update_type='xxx')
        except rpkgError:
            pass

        self.assertTrue(os.path.exists('bodhi.template.last'))


@patch.object(BugzillaClient, 'client')
class TestRequestRepo(CliTestCase):
    """Test the request-repo command"""

    def setUp(self):
        self.mock_bug = Mock()
        self.mock_bug.creator = 'Tom Hanks'
        self.mock_bug.component = 'Package Review'
        self.mock_bug.product = 'Fedora'
        self.mock_bug.assigned_to = 'Tom Brady'
        self.mock_bug.setter = 'Tom Brady'
        mod_date = Mock()
        mod_date.value = datetime.now(timezone.utc).replace(tzinfo=None).strftime('%Y%m%dT%H:%M:%S')
        self.mock_bug.flags = [{
            'status': '+',
            'name': 'fedora-review',
            'type_id': 65,
            'is_active': 1,
            'id': 1441813,
            'setter': 'Tom Brady',
            'modification_date': mod_date
        }]
        self.mock_bug.summary = ('Review Request: nethack - A rogue-like '
                                 'single player dungeon exploration game')
        super(TestRequestRepo, self).setUp()

    def get_cli(self, cli_cmd, name='fedpkg-stage', cfg='fedpkg-stage.conf',
                user_cfg='fedpkg-user-stage.conf'):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg, user_cfg=user_cfg)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo(self, mock_request_post, mock_bz):
        """Tests a standard request-repo call"""
        self.mock_bug.summary = ('Review Request: testpkg - a description')
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=True)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'testpkg', '1441813']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'backend': 'custom',
            'branch': 'rawhide',
            'bug_id': 1441813,
            'description': '',
            'distribution': 'Fedora',
            'exception': False,
            'monitor': 'monitoring',
            'namespace': 'rpms',
            'onboard_packit': 'no',
            'project_name': 'testpkg',
            'repo': 'testpkg',
            'summary': 'a description',
            'upstreamurl': ''
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_override(self, mock_request_post, mock_bz):
        """Tests a request-repo call with an overridden repo name"""
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=True)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'branch': 'rawhide',
            'bug_id': 1441813,
            'description': '',
            'exception': False,
            'monitor': 'monitoring',
            'namespace': 'rpms',
            'onboard_packit': 'no',
            'repo': 'nethack',
            'summary': ('A rogue-like single player dungeon exploration '
                        'game'),
            'upstreamurl': '',
            'backend': 'custom',
            'project_name': 'nethack',
            'distribution': 'Fedora',
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_module(self, mock_request_post, mock_bz):
        """Tests a request-repo call for a new module"""
        self.mock_bug.product = 'Fedora Modules'
        self.mock_bug.component = 'Module Review'
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=True)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', '--namespace', 'modules', 'nethack']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'branch': 'rawhide',
            'bug_id': '',
            'description': '',
            'exception': False,
            'monitor': 'monitoring',
            'namespace': 'modules',
            'onboard_packit': 'no',
            'repo': 'nethack',
            'summary': (''),
            'upstreamurl': '',
            'backend': 'custom',
            'project_name': 'nethack',
            'distribution': 'Fedora',
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_container(self, mock_request_post, mock_bz):
        """Tests a request-repo call for a new container"""
        self.mock_bug.product = 'Fedora Container Images'
        self.mock_bug.component = 'Container Review'
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=True)

        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path,
            'request-repo', '--namespace', 'container', 'nethack', '1441813'
        ]
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'branch': 'rawhide',
            'bug_id': 1441813,
            'description': '',
            'exception': False,
            'monitor': 'monitoring',
            'namespace': 'container',
            'onboard_packit': 'no',
            'repo': 'nethack',
            'summary': ('A rogue-like single player dungeon exploration '
                        'game'),
            'upstreamurl': '',
            'backend': 'custom',
            'project_name': 'nethack',
            'distribution': 'Fedora',
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_with_optional_details(
            self, mock_request_post, mock_bz):
        """Tests a request-repo call with the optional details"""
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813', '-d',
                   'a description', '-s', 'a summary', '-m', 'no-monitoring',
                   '-u', 'http://test.local', '--onboard-packit', 'pull-request']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'branch': 'rawhide',
            'bug_id': 1441813,
            'description': 'a description',
            'exception': False,
            'monitor': 'no-monitoring',
            'namespace': 'rpms',
            'onboard_packit': 'pull-request',
            'repo': 'nethack',
            'summary': 'a summary',
            'upstreamurl': 'http://test.local'
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_exception(self, mock_request_post, mock_bz):
        """Tests a request-repo call with the exception flag"""
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=True)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', '--exception', 'nethack']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'branch': 'rawhide',
            'bug_id': '',
            'description': '',
            'exception': True,
            'monitor': 'monitoring',
            'namespace': 'rpms',
            'onboard_packit': 'no',
            'repo': 'nethack',
            'summary': '',
            'upstreamurl': '',
            'backend': 'custom',
            'project_name': 'nethack',
            'distribution': 'Fedora',
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)
        # Since it is an exception, Bugzilla will not have been queried
        mock_bz.getbug.assert_not_called()

    @patch('requests.post')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_repo_monitoring_anitya_not_exist(self, mock_request_post, mock_bz):
        """Tests a request-repo call with the monitoring flag when anitya does not exist
        and arguments for creating not provided"""
        self.mock_bug.summary = ('Review Request: testpkg - a description')
        mock_bz.getbug.return_value = self.mock_bug
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv
        fedpkg.cli.does_package_exist_in_anitya = Mock(return_value=False)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'testpkg', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            "When monitoring or monitoring-with-scratch arguments are set "
            "you are also required to provide `--backend`, `--upstreamurl` "
            "and `--project-name` (if it's different from package name) arguments "
            "to create a project and package for Anitya."
        )
        try:
            cli.request_repo()
        except rpkgError as error:
            self.assertEqual(expected_error, str(error))

    def test_request_repo_wrong_package(self, mock_bz):
        """Tests request-repo errors when the package is wrong"""
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'not-nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = ('The package in the Bugzilla bug "nethack" doesn\'t '
                          'match the one provided "not-nethack"')
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_wrong_bug_product(self, mock_bz):
        """Tests request-repo errors when the bug product is not Fedora"""
        self.mock_bug.product = 'Red Hat'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'The Bugzilla bug provided is not for "Fedora" or "Fedora EPEL"'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_invalid_summary(self, mock_bz):
        """Tests request-repo errors when the bug summary has no colon"""
        self.mock_bug.summary = 'I am so wrong'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'Invalid title for this Bugzilla bug (no ":" present)'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_invalid_summary_two(self, mock_bz):
        """Tests request-repo errors when the bug summary has no dash"""
        self.mock_bug.summary = 'So:Wrong'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'Invalid title for this Bugzilla bug (no "-" present)'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_wrong_summary(self, mock_bz):
        """Tests request-repo errors when the bug summary is wrong"""
        self.mock_bug.summary = ('Review Request: fedpkg - lorum ipsum')
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = ('The package in the Bugzilla bug "fedpkg" doesn\'t '
                          'match the one provided "nethack"')
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_expired_bug(self, mock_bz):
        """Tests request-repo errors when the bug was approved over 60 days ago
        """
        self.mock_bug.flags[0]['modification_date'].value = \
            (datetime.now(timezone.utc).replace(tzinfo=None) - timedelta(days=75)).strftime(
                '%Y%m%dT%H:%M:%S')
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'The Bugzilla bug\'s review was approved over 60 days ago'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_bug_not_approved(self, mock_bz):
        """Tests request-repo errors when the bug is not approved"""
        self.mock_bug.flags[0]['name'] = 'something else'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = 'The Bugzilla bug is not approved yet'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_bug_not_assigned(self, mock_bz):
        """Tests request-repo errors when the bug is not assigned"""
        self.mock_bug.assigned_to = None
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = 'The Bugzilla bug provided is not assigned to anyone'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_repo_invalid_name(self, mock_bz):
        """Tests request-repo errors when the repo name is invalid"""
        self.mock_bug.product = 'Red Hat'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', '$nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            'The repository name "$nethack" is invalid. It must be at least '
            'two characters long with only letters, numbers, hyphens, '
            'underscores, plus signs, and/or periods. Please note that the '
            'project cannot start with a period or a plus sign.')
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch('requests.post')
    def test_request_repo_pagure_error(self, mock_request_post, mock_bz):
        """Tests a standard request-repo call when the Pagure API call fails"""
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack', '1441813']
        cli = self.get_cli(cli_cmd)
        mock_rv = Mock()
        mock_rv.ok = False
        mock_rv.json.return_value = {'error': 'some error'}
        mock_request_post.return_value = mock_rv
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            expected_error = ('The following error occurred while creating a '
                              'new issue in Pagure: some error')
            self.assertEqual(str(error), expected_error)

    def test_request_repo_no_bug(self, mock_bz):
        """Tests request-repo errors when no bug or exception is provided"""
        self.mock_bug.product = 'Red Hat'
        mock_bz.getbug.return_value = self.mock_bug
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', 'nethack']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'A Bugzilla bug is required on new repository requests'
        try:
            cli.request_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch('requests.post')
    def test_request_repo_without_initial_commit(
            self, mock_request_post, mock_bz):
        """Tests a request-repo call with --no-initial-commit"""

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-repo', '--no-initial-commit', '--exception',
                   'somepkg']
        cli = self.get_cli(cli_cmd)
        cli.request_repo()

        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertIn('initial_commit', actual_issue_content)
        self.assertFalse(actual_issue_content['initial_commit'])


class TestRequestBranch(CliTestCase):
    """Test the request-branch command"""

    def setUp(self):
        super(TestRequestBranch, self).setUp()

    def tearDown(self):
        super(TestRequestBranch, self).tearDown()

    def get_cli(self, cli_cmd, name='fedpkg-stage', cfg='fedpkg-stage.conf',
                user_cfg='fedpkg-user-stage.conf'):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg, user_cfg=user_cfg)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch(self, mock_grb, mock_request_post, mock_request_get):
        """Tests request-branch"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        # Checkout the f27 branch
        self.run_cmd(['git', 'checkout', 'f27'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(1, mock_request_get.call_count)

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'testpkg',
            'namespace': 'rpms',
            'branch': 'f27',
            'create_git_branch': True
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_existing_branch(self, mock_grb, mock_request_post, mock_request_get):
        """Tests request-existing-branch"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        # branch 'f27' already exists and there is no POST request sent to Pagure (it would create
        # an issue)
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "f27", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        # Checkout the f27 branch
        self.run_cmd(['git', 'checkout', 'f27'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(1, mock_request_get.call_count)
        mock_request_post.assert_not_called()

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_override(self, mock_grb, mock_request_post, mock_request_get):
        """Tests request-branch with an overridden package and branch name"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch', '--repo', 'nethack', 'f27']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(1, mock_request_get.call_count)

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'nethack',
            'namespace': 'rpms',
            'branch': 'f27',
            'create_git_branch': True
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_epel_branch_override(
        self, mock_grb, mock_request_post, mock_request_get
    ):
        """Tests request-epel-branch-override"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7', 'epel8']}
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"arches": [], "packages": {}}
        mock_request_get.return_value = mock_rv
        # Checkout the epel7 branch
        self.run_cmd(['git', 'checkout', 'epel7'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch', '--repo', 'sudoku', 'epel8']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(2, mock_request_get.call_count)

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'sudoku',
            'namespace': 'rpms',
            'branch': 'epel8',
            'create_git_branch': True
        }
        self.assertEqual(len(mock_request_post.call_args_list), 1)

        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)

        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_module(self, mock_grb, mock_request_post, mock_request_get):
        """Tests request-branch for a new module branch"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch',
                   '--repo', 'nethack', '--namespace', 'modules', 'f27']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'nethack',
            'namespace': 'modules',
            'branch': 'f27',
            'create_git_branch': True
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    def assert_request_branch_container(self, cli_cmd, mock_grb, mock_request_post,
                                        mock_request_get):
        """Tests request-branch for a new container branch"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'nethack',
            'namespace': 'container',
            'branch': 'f27',
            'create_git_branch': True
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)

    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_with_global_option_name_and_namespace(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', '--namespace', 'container',
                   'request-branch', 'f27']
        self.assert_request_branch_container(cli_cmd)

    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_with_its_own_option_repo_and_namespace(self):
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch',
                   '--repo', 'nethack', '--namespace', 'container', 'f27']
        self.assert_request_branch_container(cli_cmd)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('fedpkg.cli.verify_sls')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_sls(self, mock_verify_sls, mock_grb,
                                mock_request_post, mock_request_get):
        """Tests request-branch with service levels"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        responses = []
        mock_rv_get = Mock()
        mock_rv_get.ok = True
        mock_rv_get.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        responses.append(mock_rv_get)
        mock_rv_get = Mock()
        mock_rv_get.ok = False
        mock_rv_get.json.return_value = {'error': 'Project not found', 'error_code': 'ENOPROJECT'}
        responses.append(mock_rv_get)
        mock_request_get.side_effect = responses

        responses = []
        for idx in range(2, 5):
            mock_rv_post = Mock()
            mock_rv_post.ok = True
            mock_rv_post.json.return_value = {'issue': {'id': idx}}
            responses.append(mock_rv_post)
        mock_request_post.side_effect = responses

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch', '9', '--sl',
                   'security_fixes:2030-12-01', 'bug_fixes:2030-12-01']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(2, mock_request_get.call_count)
        # Get the data that was submitted to Pagure
        output = sys.stdout.getvalue().strip()
        # Three bugs are filed.  One for the rpm branch, and one for a new
        # module repo, and one for the matching module branch.
        expected_output = (
            'https://pagure.stg.example.com/releng/'
            'fedora-scm-requests/issue/2\n'
            'https://pagure.stg.example.com/releng/'
            'fedora-scm-requests/issue/3\n'
            'https://pagure.stg.example.com/releng/'
            'fedora-scm-requests/issue/4'
        )
        self.assertMultiLineEqual(output, expected_output)

        # Check for rpm branch..
        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'nethack',
            'namespace': 'rpms',
            'branch': '9',
            'create_git_branch': True,
            'sls': {
                'security_fixes': '2030-12-01',
                'bug_fixes': '2030-12-01'
            }
        }
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertDictEqual(expected_issue_content, actual_issue_content)

        # Check for the module repo request..
        summary = u'Automatically requested module for rpms/nethack:9.'
        expected_issue_content = {
            u'action': u'new_repo',
            u'branch': u'rawhide',
            u'bug_id': u'',
            u'description': summary,
            u'exception': True,
            u'monitor': u'no-monitoring',
            u'namespace': u'modules',
            u'onboard_packit': u'no',
            u'repo': u'nethack',
            u'summary': summary,
            u'upstreamurl': u''
        }
        post_data = mock_request_post.call_args_list[1][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertDictEqual(expected_issue_content, actual_issue_content)

        # Check for module branch..
        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'nethack',
            'namespace': 'modules',
            'branch': '9',
            'create_git_branch': True,
            'sls': {
                'security_fixes': '2030-12-01',
                'bug_fixes': '2030-12-01'
            }
        }
        post_data = mock_request_post.call_args_list[2][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertDictEqual(expected_issue_content, actual_issue_content)

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_branch_all_releases(self, mock_grb, mock_request_post, mock_request_get):
        """Tests request-branch with the '--all-releases' option """
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        get_side_effect = []
        for i in range(1, 4):
            mock_rv = Mock()
            mock_rv.ok = True
            mock_rv.json.return_value = {"branches": ["main", "rawhide"]}
            get_side_effect.append(mock_rv)
        mock_request_post.side_effect = get_side_effect

        post_side_effect = []
        for i in range(1, 4):
            mock_rv = Mock()
            mock_rv.ok = True
            mock_rv.json.return_value = {'issue': {'id': i}}
            post_side_effect.append(mock_rv)
        mock_request_post.side_effect = post_side_effect

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch',
                   '--all-releases']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        self.assertEqual(3, mock_request_get.call_count)

        for i in range(3):
            expected_issue_content = {
                'action': 'new_branch',
                'repo': 'nethack',
                'namespace': 'rpms',
                'branch': 'f' + str(27 - i),
                'create_git_branch': True
            }
            post_data = mock_request_post.call_args_list[i][1]['data']
            actual_issue_content = json.loads(json.loads(
                post_data)['issue_content'].strip('```'))
            self.assertEqual(expected_issue_content, actual_issue_content)

        output = sys.stdout.getvalue().strip()
        expected_output = """\
https://pagure.stg.example.com/releng/fedora-scm-requests/issue/1
https://pagure.stg.example.com/releng/fedora-scm-requests/issue/2
https://pagure.stg.example.com/releng/fedora-scm-requests/issue/3"""
        self.assertEqual(output, expected_output)

    def test_request_branch_invalid_use_of_all_releases(self):
        """Tests request-branch with a branch and the '--all-releases' option
        """
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch', 'f27',
                   '--all-releases']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'You cannot specify a branch with the "--all-releases" option'
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_branch_invalid_use_of_all_releases_sl(self):
        """Tests request-branch with an SL and the '--all-releases' option
        """
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch',
                   '--all-releases', '--sl', 'security_fixes:2020-01-01']
        cli = self.get_cli(cli_cmd)
        expected_error = ('You cannot specify service levels with the '
                          '"--all-releases" option')
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch('fedpkg.cli.get_release_branches')
    def test_request_branch_invalid_sls(self, mock_grb):
        """Tests request-branch with invalid service levels"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch', '9', '--sl',
                   'security_fixes-2030-12-01', 'bug_fixes:2030-12-01']
        cli = self.get_cli(cli_cmd)
        expected_error = \
            'The SL "security_fixes-2030-12-01" is in an invalid format'
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch('fedpkg.cli.get_release_branches')
    def test_request_branch_sls_on_release_branch_error(self, mock_grb):
        """Tests request-branch with a release branch and service levels"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', 'request-branch', 'f27', '--sl',
                   'security_fixes-2030-12-01', 'bug_fixes:2030-12-01']
        cli = self.get_cli(cli_cmd)
        expected_error = 'You can\'t provide SLs for release branches'
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_branch_invalid_module_branch_name(self):
        """Test request-branch raises an exception when a invalid module branch
        name is supplied"""
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', '--namespace', 'modules',
                   'request-branch', 'some:branch']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            'Only characters, numbers, periods, dashes, underscores, and '
            'pluses are allowed in module branch names')
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_branch_invalid_flatpak_branch_name(self):
        """Test request-branch raises an exception when a invalid module branch
        name is supplied"""
        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'nethack', '--namespace', 'flatpaks',
                   'request-branch', 'some:branch']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            'Only characters, numbers, periods, dashes, underscores, and '
            'pluses are allowed in flatpak branch names')
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    def test_request_branch_no_branch(self):
        """Test request-branch raises an exception when a branch isn't supplied
        """
        tempdir = mkdtemp()
        cli_cmd = ['fedpkg-stage', '--path', tempdir,
                   '--name', 'nethack', 'request-branch']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            'You must specify a branch if you are not in a git repository')
        try:
            cli.request_branch()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)
        finally:
            rmdir(tempdir)

    @patch('requests.get')
    def test_request_branch_invalid_epel_package(self, mock_get):
        """Test request-branch raises an exception when an EPEL branch is
        requested but this package is already an EL package on all supported
        arches"""
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {
            'arches': ['noarch', 'x86_64', 'i686', 'ppc64', 'ppc', 'ppc64le'],
            'packages': {
                'kernel': {'arch': [
                    'noarch', 'x86_64', 'ppc64', 'ppc64le']},
                'glibc': {'arch': [
                    'i686', 'x86_64', 'ppc', 'ppc64', 'ppc64le']}
            }
        }
        mock_get.return_value = mock_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   '--name', 'kernel', 'request-branch', 'epel7']
        cli = self.get_cli(cli_cmd)
        expected_error = (
            'This package is already an EL package and is built on all '
            'supported arches, therefore, it cannot be in EPEL. If this is a '
            'mistake or you have an exception, please contact the Release '
            'Engineering team.')
        with self.assertRaisesRegex(rpkgError, expected_error):
            cli.request_branch()

    @patch('requests.get')
    @patch('requests.post')
    @patch('fedpkg.cli.get_release_branches')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_with_repo_option(self, mock_grb, mock_request_post, mock_request_get):
        """Test request branch with option --repo"""
        mock_grb.return_value = {'fedora': ['f25', 'f26', 'f27'],
                                 'epel': ['el6', 'epel7']}

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"branches": ["f25", "f26", "main", "rawhide"]}
        mock_request_get.return_value = mock_rv

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 2}}
        mock_request_post.return_value = mock_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-branch', '--repo', 'foo', 'f27']
        cli = self.get_cli(cli_cmd)
        cli.request_branch()

        expected_issue_content = {
            'action': 'new_branch',
            'repo': 'foo',
            'namespace': 'rpms',
            'branch': 'f27',
            'create_git_branch': True
        }
        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/2')
        self.assertEqual(output, expected_output)


class TestRequestTestsRepo(CliTestCase):
    """Test the request-tests-repo command"""

    def setUp(self):
        super(TestRequestTestsRepo, self).setUp()

    def tearDown(self):
        super(TestRequestTestsRepo, self).tearDown()

    def get_cli(self, cli_cmd, name='fedpkg-stage', cfg='fedpkg-stage.conf',
                user_cfg='fedpkg-user-stage.conf'):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg, user_cfg=user_cfg)

    @patch('requests.post')
    @patch('requests.get')
    @patch('sys.stdout', new=io.StringIO())
    def test_request_tests_repo(self, mock_request_get, mock_request_post):
        """Tests request-tests-repo"""

        # mock the request.get from assert_new_tests_repo
        mock_get_rv = Mock()
        mock_get_rv.ok = False
        mock_request_get.return_value = mock_get_rv

        # mock the request.post to Pagure
        mock_post_rv = Mock()
        mock_post_rv.ok = True
        mock_post_rv.json.return_value = {'issue': {'id': 1}}
        mock_request_post.return_value = mock_post_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-tests-repo', 'foo', 'Some description']
        cli = self.get_cli(cli_cmd)
        cli.request_tests_repo()

        expected_issue_content = {
            'action': 'new_repo',
            'backend': 'custom',
            'branch': 'main',
            'bug_id': '',
            'monitor': 'no-monitoring',
            'namespace': 'tests',
            'onboard_packit': 'no',
            'repo': 'foo',
            'description': 'Some description',
            'distribution': 'Fedora',
            'upstreamurl': '',
            'project_name': 'foo',
        }

        # Get the data that was submitted to Pagure
        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)

        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/1')
        self.assertEqual(output, expected_output)

    @patch('requests.get')
    def test_request_tests_repo_exists(self, mock_request_get):
        """Tests request-tests-repo exception if repo already exists"""

        # mock the request.get from assert_new_tests_repo
        mock_get_rv = Mock()
        mock_get_rv.ok = True
        mock_request_get.return_value = mock_get_rv

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                   'request-tests-repo', 'foo', 'Some description']
        cli = self.get_cli(cli_cmd)

        expected_error = (
            'Repository git://pkgs.stg.example.com/tests/foo already exists')

        try:
            cli.request_tests_repo()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)


class TestCheckBodhiVersion(unittest.TestCase):
    """Test check_bodhi_version"""

    @patch('fedpkg.cli.distribution')
    def test_no_6_x_version_installed(self, mock_distribution):
        mock_distribution.return_value.version = '5.7.5'
        self.assertRaisesRegex(
            rpkgError, r'bodhi-client < 6\.0\.0 is not supported\.',
            check_bodhi_version)


@unittest.skipUnless(bodhi, 'Skip if no supported bodhi-client is available')
class TestBodhiOverride(CliTestCase):
    """Test command override"""

    def setUp(self):
        super(TestBodhiOverride, self).setUp()

        self.cbv_p = patch('fedpkg.cli.check_bodhi_version')
        self.cbv_m = self.cbv_p.start()

        self.anon_kojisession_p = patch('fedpkg.Commands.anon_kojisession',
                                        new_callable=PropertyMock)
        self.anon_kojisession_m = self.anon_kojisession_p.start()
        self.kojisession = self.anon_kojisession_m.return_value

        self.tempdir = mkdtemp()
        self.os_environ_patcher = patch.dict('os.environ', {'EDITOR': 'vi', 'HOME': self.tempdir})
        self.os_environ_patcher.start()

        self.oidcmeta_patcher = patch(
            'bodhi.client.oidcclient.OIDCClient._get_provider_metadata', _mock_metadata
        )
        self.oidcmeta_patcher.start()

        # Fake build returned from Koji for the specified build NVR in tests
        self.kojisession.getBuild.return_value = {'build_id': 1}

    def tearDown(self):
        self.anon_kojisession_p.stop()
        self.cbv_p.stop()
        self.os_environ_patcher.stop()
        self.oidcmeta_patcher.stop()
        rmdir(self.tempdir)
        super(TestBodhiOverride, self).tearDown()

    def test_raise_error_if_build_not_exist(self):
        self.kojisession.getBuild.return_value = None

        build_nvr = 'rpkg-1.54-1.fc28'
        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'create',
            '--duration', '7', '--notes', 'build for fedpkg',
            build_nvr
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            self.assertRaisesRegex(
                rpkgError, 'Build {0} does not exist'.format(build_nvr),
                cli.create_buildroot_override)

        self.kojisession.getBuild.assert_called_once_with(build_nvr)

    @patch('bodhi.client.bindings.BodhiClient.list_overrides')
    @patch('bodhi.client.bindings.BodhiClient.save_override')
    @patch('bodhi.client.bindings.BodhiClient.override_str')
    def test_create_for_given_build(
            self, override_str, save_override, list_overrides):
        list_overrides.return_value = {'total': 0}
        expiration_date = datetime.now() + timedelta(days=7)
        new_override = {
            'expiration_date': expiration_date.strftime('%Y-%m-%d %H:%M:%S'),
            'notes': 'build for fedpkg'
        }
        save_override.return_value = new_override

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'create',
            '--duration', '7', '--notes', 'build for fedpkg',
            'rpkg-1.54-1.fc28'
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            with patch.object(cli.cmd, 'log') as log:
                cli.create_buildroot_override()

                override_str.assert_called_once_with(
                    save_override.return_value,
                    minimal=False)
                log.info.assert_any_call(
                    override_str.return_value)

        save_override.assert_called_once_with(
            nvr='rpkg-1.54-1.fc28',
            duration=7,
            notes='build for fedpkg')

    @patch('fedpkg.BodhiClient')
    @patch('fedpkg.Commands.nvr', new_callable=PropertyMock)
    def test_create_from_current_branch(self, nvr, BodhiClient):
        nvr.return_value = 'rpkg-1.54-2.fc28'
        bodhi_client = BodhiClient.return_value
        bodhi_client.list_overrides.return_value = {'total': 0}

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'create',
            '--duration', '7', '--notes', 'build for fedpkg',
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli.create_buildroot_override()

        bodhi_client.save_override.assert_called_once_with(
            nvr='rpkg-1.54-2.fc28',
            duration=7,
            notes='build for fedpkg')

    @patch('fedpkg.BodhiClient')
    @patch('fedpkg.Commands.nvr', new_callable=PropertyMock)
    def test_override_already_exists_but_expired(self, nvr, BodhiClient):
        nvr.return_value = 'rpkg-1.54-2.fc28'
        today = datetime.today()
        fake_expiration_date = today - timedelta(days=10)
        BodhiClient.return_value.list_overrides.return_value = {
            'total': 1,
            'overrides': [{
                'expiration_date': fake_expiration_date.strftime(
                    '%Y-%m-%d %H:%M:%S')
            }]
        }

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'create',
            '--duration', '7', '--notes', 'build for fedpkg',
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            with patch.object(cli.cmd, 'log') as log:
                cli.create_buildroot_override()
                log.info.assert_any_call(
                    'Buildroot override for %s exists and is expired. Consider'
                    ' using command `override extend` to extend duration.',
                    'rpkg-1.54-2.fc28')

    @patch('fedpkg.BodhiClient')
    @patch('fedpkg.Commands.nvr', new_callable=PropertyMock)
    def test_override_already_exists_but_not_expired(self, nvr, BodhiClient):
        nvr.return_value = 'rpkg-1.54-2.fc28'
        today = datetime.today()
        fake_expiration_date = today + timedelta(days=10)
        BodhiClient.return_value.list_overrides.return_value = {
            'total': 1,
            'overrides': [{
                'expiration_date': fake_expiration_date.strftime(
                    '%Y-%m-%d %H:%M:%S')
            }]
        }

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'create',
            '--duration', '7', '--notes', 'build for fedpkg',
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            with patch.object(cli.cmd, 'log') as log:
                cli.create_buildroot_override()
                log.info.assert_any_call(
                    'Buildroot override for %s already exists and not '
                    'expired.', 'rpkg-1.54-2.fc28')

    def test_invalid_duration_option(self):
        cli_cmds = (
            (
                [
                    'fedpkg', '--path', self.cloned_repo_path,
                    'override', 'create',
                    '--duration', 'abc', '--notes', 'build for fedpkg',
                ],
                'duration must be an integer'
            ),
            (
                [
                    'fedpkg', '--path', self.cloned_repo_path,
                    'override', 'create',
                    '--duration', '0', '--notes', 'build for fedpkg',
                ],
                'override should have 1 day to exist at least'
            )
        )

        for cmd, expected_output in cli_cmds:
            with patch('sys.argv', new=cmd):
                with patch('sys.stderr', new=io.StringIO()):
                    with self.assertRaises(SystemExit):
                        self.new_cli()
                    output = sys.stderr.getvalue()
                    self.assertIn(expected_output, output)


@unittest.skipUnless(bodhi, 'Skip if no supported bodhi-client is available')
class TestBodhiOverrideExtend(CliTestCase):
    """Test command `override extend`"""

    def setUp(self):
        super(TestBodhiOverrideExtend, self).setUp()

        # No need to check bodhi version for tests
        self.cbv_p = patch('fedpkg.cli.check_bodhi_version')
        self.cbv_p.start()

        self.anon_kojisession_p = patch('fedpkg.Commands.anon_kojisession',
                                        new_callable=PropertyMock)
        self.anon_kojisession_m = self.anon_kojisession_p.start()
        self.kojisession = self.anon_kojisession_m.return_value

        self.tempdir = mkdtemp()
        self.os_environ_patcher = patch.dict('os.environ', {'EDITOR': 'vi', 'HOME': self.tempdir})
        self.os_environ_patcher.start()

        self.oidcmeta_patcher = patch(
            'bodhi.client.oidcclient.OIDCClient._get_provider_metadata', _mock_metadata
        )
        self.oidcmeta_patcher.start()

        # Fake build returned from Koji for the specified build NVR in tests
        self.kojisession.getBuild.return_value = {'build_id': 1}

    def tearDown(self):
        self.anon_kojisession_p.stop()
        self.cbv_p.stop()
        self.os_environ_patcher.stop()
        self.oidcmeta_patcher.stop()
        rmdir(self.tempdir)
        super(TestBodhiOverrideExtend, self).tearDown()

    def test_specified_build_not_exist(self):
        self.kojisession.getBuild.return_value = None

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '7', 'somepkg-1.54-2.fc28'
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            self.assertRaisesRegex(
                rpkgError, 'Build somepkg-1.54-2.fc28 does not exist.',
                cli.extend_buildroot_override)

    @patch('fedpkg.BodhiClient.list_overrides')
    def test_no_override_for_build(self, list_overrides):
        list_overrides.return_value = {'total': 0}

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '7', 'somepkg-1.54-2.fc28'
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            with patch.object(cli.cmd, 'log') as log:
                cli.extend_buildroot_override()
                log.info.assert_any_call('No buildroot override for build %s',
                                         'somepkg-1.54-2.fc28')

    @patch('fedpkg.BodhiClient.list_overrides')
    @patch('fedpkg.BodhiClient.csrf')
    @patch('fedpkg.BodhiClient.send_request')
    def test_extend_override_by_days(
            self, send_request, csrf, list_overrides):
        utcnow = datetime.now(timezone.utc).replace(tzinfo=None)
        override_expiration_date = utcnow + timedelta(days=7)
        build_nvr = 'somepkg-1.54-2.fc28'
        build_override = {
            'expiration_date': override_expiration_date.strftime('%Y-%m-%d %H:%M:%S'),
            'nvr': build_nvr,
            'notes': 'build for other package',
            'build': {'nvr': build_nvr},
            'submitter': {'name': 'someone'},
            'expired_date': utcnow - timedelta(days=20)
        }

        list_overrides.return_value = {
            'total': 1,
            'overrides': [build_override]
        }
        edited_override = build_override.copy()
        expected_expiration_date = override_expiration_date + timedelta(days=2)
        edited_override['expiration_date'] = \
            expected_expiration_date.strftime('%Y-%m-%d %H:%M:%S')
        send_request.return_value = edited_override

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '2', build_nvr
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli.extend_buildroot_override()

        # Ensure no microsecond is included in the expected expiration data
        new_date = override_expiration_date + timedelta(days=2)
        expected_expiration_date = datetime(year=new_date.year,
                                            month=new_date.month,
                                            day=new_date.day,
                                            hour=new_date.hour,
                                            minute=new_date.minute,
                                            second=new_date.second)
        request_data = {
            'expiration_date': expected_expiration_date,
            'nvr': build_nvr,
            'notes': build_override['notes'],
            'csrf_token': csrf.return_value,
        }
        send_request.assert_called_once_with(
            'overrides/', verb='POST', auth=True, data=request_data)

    @patch('fedpkg.BodhiClient.list_overrides')
    @patch('fedpkg.BodhiClient.csrf')
    @patch('fedpkg.BodhiClient.send_request')
    def test_extend_override_by_specific_date(
            self, send_request, csrf, list_overrides):
        utcnow = datetime.now(timezone.utc)
        override_expiration_date = utcnow + timedelta(days=7)
        build_nvr = 'somepkg-1.54-2.fc28'
        build_override = {
            'expiration_date': override_expiration_date.strftime('%Y-%m-%d %H:%M:%S'),
            'nvr': build_nvr,
            'notes': 'build for other package',
            'build': {'nvr': build_nvr},
            'submitter': {'name': 'someone'},
            'expired_date': utcnow - timedelta(days=20)
        }

        list_overrides.return_value = {
            'total': 1,
            'overrides': [build_override]
        }

        new_expiration_date = override_expiration_date + timedelta(days=4)
        # Ensure no microsecond is included in the expected expiration data
        expected_expiration_date = datetime(
            year=new_expiration_date.year,
            month=new_expiration_date.month,
            day=new_expiration_date.day,
            hour=override_expiration_date.hour,
            minute=override_expiration_date.minute,
            second=override_expiration_date.second)

        edited_override = build_override.copy()
        edited_override['expiration_date'] = \
            expected_expiration_date.strftime('%Y-%m-%d %H:%M:%S')
        send_request.return_value = edited_override

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', new_expiration_date.strftime('%Y-%m-%d'),
            build_nvr
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli.extend_buildroot_override()

        request_data = {
            'expiration_date': expected_expiration_date,
            'nvr': build_nvr,
            'notes': build_override['notes'],
            'csrf_token': csrf.return_value,
        }
        send_request.assert_called_once_with(
            'overrides/', verb='POST', auth=True, data=request_data)

    @patch('fedpkg.BodhiClient.list_overrides')
    @patch('fedpkg.BodhiClient.csrf')
    @patch('fedpkg.BodhiClient.send_request')
    @freeze_time('2018-06-08 16:17:30')
    def test_extend_for_expired_override(
            self, send_request, csrf, list_overrides):
        utcnow = datetime.now(timezone.utc).replace(tzinfo=None)
        # Make override expired
        override_expiration_date = utcnow - timedelta(days=7)
        build_nvr = 'somepkg-1.54-2.fc28'
        build_override = {
            'expiration_date': override_expiration_date.strftime('%Y-%m-%d %H:%M:%S'),
            'nvr': build_nvr,
            'notes': 'build for other package',
            'build': {'nvr': build_nvr},
            'submitter': {'name': 'someone'},
            'expired_date': utcnow - timedelta(days=20)
        }

        list_overrides.return_value = {
            'total': 1,
            'overrides': [build_override]
        }

        # Ensure no microsecond is included in the expected expiration data
        new_date = utcnow + timedelta(days=2)
        expected_expiration_date = datetime(
            year=new_date.year,
            month=new_date.month,
            day=new_date.day,
            hour=new_date.hour,
            minute=new_date.minute,
            second=new_date.second)

        edited_override = build_override.copy()
        edited_override['expiration_date'] = \
            expected_expiration_date.strftime('%Y-%m-%d %H:%M:%S')
        send_request.return_value = edited_override

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '2', build_nvr
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli.extend_buildroot_override()

        request_data = {
            'expiration_date': expected_expiration_date,
            'nvr': build_nvr,
            'notes': build_override['notes'],
            'csrf_token': csrf.return_value,
        }
        send_request.assert_called_once_with(
            'overrides/', verb='POST', auth=True, data=request_data)

    @patch('fedpkg.BodhiClient.list_overrides')
    @patch('fedpkg.BodhiClient.csrf')
    @patch('fedpkg.BodhiClient.extend_override')
    def test_error_handled_properly_when_fail_to_request(
            self, extend_override, csrf, list_overrides):
        extend_override.side_effect = Exception

        utcnow = datetime.now(timezone.utc).replace(tzinfo=None)
        override_expiration_date = utcnow + timedelta(days=7)
        build_nvr = 'somepkg-1.54-2.fc28'

        list_overrides.return_value = {
            'total': 1,
            'overrides': [{
                'expiration_date': override_expiration_date.strftime('%Y-%m-%d %H:%M:%S'),
                'nvr': build_nvr,
                'notes': 'build for other package',
                'build': {'nvr': build_nvr},
                'submitter': {'name': 'someone'},
                'expired_date': utcnow - timedelta(days=20)
            }]
        }

        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '2', build_nvr
        ]

        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            self.assertRaisesRegex(rpkgError, '',
                                   cli.extend_buildroot_override)

    @freeze_time('2018-07-22')
    @patch('fedpkg.BodhiClient.list_overrides')
    def test_raise_error_if_duration_less_than_today(self, list_overrides):
        build_nvr = 'somepkg-1.54-2.fc28'
        build_override = {
            'expiration_date': '2018-03-01 12:12:12',
            'nvr': build_nvr,
            'notes': 'build for other package',
            'build': {'nvr': build_nvr},
            'submitter': {'name': 'someone'},
            'expired_date': '2018-03-01 12:12:12',
        }

        list_overrides.return_value = {
            'total': 1,
            'overrides': [build_override]
        }

        # This duration should cause the expected error.
        duration = '2018-07-18'
        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', duration, build_nvr
        ]
        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            self.assertRaisesRegex(
                rpkgError,
                'specified expiration date .+ should be future date',
                cli.extend_buildroot_override)

    def test_invalid_duration_of_concrete_date_format(self):
        cli_cmd = [
            'fedpkg', '--path', self.cloned_repo_path,
            'override', 'extend', '2019/01/10', 'rpkg-1.10-1.fc28'
        ]
        with patch('sys.argv', new=cli_cmd):
            with patch('sys.stderr', new=io.StringIO()):
                with self.assertRaises(SystemExit):
                    self.new_cli()
                output = sys.stderr.getvalue()
                self.assertIn('Invalid expiration date', output)


class TestReadReleasesFromLocalConfig(CliTestCase):
    """Test read releases from local config file"""

    def setUp(self):
        super(TestReadReleasesFromLocalConfig, self).setUp()
        self.active_releases = {
            'fedora': ['f28', 'f27'],
            'epel': ['el6', 'epel7'],
        }
        self.fake_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']

        self.package_cfg = os.path.join(self.cloned_repo_path,
                                        fedpkg.cli.LOCAL_PACKAGE_CONFIG)
        self.write_file(self.package_cfg,
                        content='[koji]\ntargets=rawhide f28 fedora epel')

    @patch('pyrpkg.utils.validate_path')
    def test_no_config_file_is_create(self, validate_path):
        error_msg = 'given path \'{0}\' doesn\'t exist'.format(self.cloned_repo_path)
        validate_path.side_effect = argparse.ArgumentTypeError(error_msg)
        with patch('sys.argv', new=self.fake_cmd):
            with patch('sys.stderr', new=io.StringIO()):
                # argparse.ArgumentTypeError turns to SystemExit
                with self.assertRaises(SystemExit):
                    self.new_cli()
                validate_path.assert_called_once_with(self.cloned_repo_path)
                output = sys.stderr.getvalue().strip()
                self.assertIn(error_msg, output)

    def test_no_build_target_is_configured(self):
        with patch('sys.argv', new=self.fake_cmd):
            cli = self.new_cli()

        self.write_file(self.package_cfg, content='[koji]')

        rels = cli.read_releases_from_local_config(self.active_releases)
        self.assertIsNone(rels)

    def test_config_file_is_not_accessible(self):
        with patch('sys.argv', new=self.fake_cmd):
            cli = self.new_cli()

        with patch('fedpkg.cli.configparser.ConfigParser.read') as read:
            read.return_value = []

            self.assertRaisesRegex(
                rpkgError, '.+ not accessible',
                cli.read_releases_from_local_config, self.active_releases)

    def test_get_expanded_releases(self):
        with patch('sys.argv', new=self.fake_cmd):
            cli = self.new_cli()

        rels = cli.read_releases_from_local_config(self.active_releases)
        rels = sorted(rels)
        self.assertEqual(['el6', 'epel7', 'f27', 'f28', 'rawhide'], rels)


class TestIsStreamBranch(CliTestCase):
    """Test fedpkgClient.is_stream_branch"""

    def setUp(self):
        super(TestIsStreamBranch, self).setUp()
        self.fake_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']

    def test_not_a_stream_branch(self):
        with patch('sys.argv', new=self.fake_cmd):
            cli = self.new_cli()

        result = cli.is_stream_branch(
            [{'name': '8', 'active': True}, {'name': '10', 'active': True}],
            'f28')
        self.assertFalse(result)

    def test_branch_is_stream_branch(self):
        with patch('sys.argv', new=self.fake_cmd):
            cli = self.new_cli()

        result = cli.is_stream_branch(
            ['epel8', 'epel9'], 'epel8')
        self.assertTrue(result)


class TestBuildFromStreamBranch(CliTestCase):
    """Test build command to build from stream branch"""

    @patch('pyrpkg.cli.cliClient._build')
    @patch('fedpkg.cli.get_stream_branches')
    def test_build_as_normal_if_branch_is_not_stream_branch(
            self, get_stream_branches, _build):
        get_stream_branches.return_value = [{'name': '8', 'active': True}]

        self.checkout_branch(git.Repo(self.cloned_repo_path), 'f27')

        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']
        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli._build()

        _build.assert_called_once_with(None)

    @patch('pyrpkg.cli.cliClient._build')
    @patch('fedpkg.cli.get_stream_branches')
    @patch('fedpkg.cli.get_release_branches')
    def test_build_as_normal_if_no_config_file_is_create(
            self, get_release_branches, get_stream_branches, _build):
        get_release_branches.return_value = {
            'fedora': ['f28', 'f27'],
            'epel': ['el6', 'epel7'],
        }
        get_stream_branches.return_value = [{'name': '8', 'active': True}]
        self.checkout_branch(git.Repo(self.cloned_repo_path), '8')

        # There is no config file created originally. So, nothing to do here
        # to run this test.

        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']
        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli._build()

        _build.assert_called_once_with(None)

    @patch('pyrpkg.cli.cliClient._build')
    @patch('fedpkg.cli.get_stream_branches')
    @patch('fedpkg.cli.get_release_branches')
    def test_build_as_normal_if_no_build_target_is_configured(
            self, get_release_branches, get_stream_branches, _build):
        get_release_branches.return_value = {
            'fedora': ['f28', 'f27'],
            'epel': ['el6', 'epel7'],
        }
        get_stream_branches.return_value = [{'name': '8', 'active': True}]
        self.checkout_branch(git.Repo(self.cloned_repo_path), '8')

        # Create local config file without option targets for this test
        self.write_file(
            os.path.join(self.cloned_repo_path,
                         fedpkg.cli.LOCAL_PACKAGE_CONFIG),
            content='[koji]')

        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']
        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            cli._build()

        _build.assert_called_once_with(None)

    @patch('distro.os_release_info', return_value={'variant': 'non-ELN'})
    @patch('pyrpkg.cli.cliClient._build')
    @patch('fedpkg.Commands.build_target')
    @patch('fedpkg.cli.get_stream_branches')
    @patch('fedpkg.cli.get_release_branches')
    def test_submit_builds(
            self, get_release_branches, get_stream_branches, build_target,
            _build, os_release_info):
        get_release_branches.return_value = {
            'fedora': ['f28', 'f27'],
            'epel': ['epel7', 'epel8', 'epel9'],
        }
        get_stream_branches.return_value = ['8', 'epel9']
        _build.side_effect = [1, 2]
        self.checkout_branch(git.Repo(self.cloned_repo_path), '8')

        # Create local config file without option targets for this test
        self.write_file(
            os.path.join(self.cloned_repo_path,
                         fedpkg.cli.LOCAL_PACKAGE_CONFIG),
            content='[koji]\ntargets = fedora')

        cli_cmd = ['fedpkg', '--path', self.cloned_repo_path, 'build']
        with patch('sys.argv', new=cli_cmd):
            cli = self.new_cli()
            task_ids = cli._build()

        build_target.assert_has_calls([call('f27'), call('f28')])
        _build.assert_has_calls([call(None), call(None)])
        self.assertEqual([1, 2], task_ids)


@patch('fedpkg.cli.get_release_branches',
       return_value={
           'epel': ['el6', 'epel7'],
           'fedora': ['f29', 'f28']
       })
class TestReleasesInfo(CliTestCase):
    """Test command releases-info"""

    require_test_repos = False

    def assert_output_releases(self, expected_output, option=[]):
        with patch('sys.argv', ['fedpkg', 'releases-info'] + option):
            cli = self.new_cli()
            with patch('sys.stdout', new=io.StringIO()):
                cli.show_releases_info()
                output = sys.stdout.getvalue().strip()
                self.assertEqual(expected_output, output)

    def test_print_epel_releases_only(self, mock_grb):
        self.assert_output_releases('el6 epel7', option=['--epel'])

    def test_print_fedora_releases_only(self, mock_grb):
        self.assert_output_releases('f29 f28', option=['--fedora'])

    def test_print_joined_releaes(self, mock_grb):
        self.assert_output_releases('f29 f28 el6 epel7', option=['--join'])

    def test_print_releases_in_default(self, mock_grb):
        self.assert_output_releases('Fedora: f29 f28\nEPEL: el6 epel7')


class TestRetire(CliTestCase):
    """
    Test retire operation with additional fedpkg Fedora release checking
    """

    @patch('requests.get')
    def retire_release(self, branch, release_state, mock_get):
        mock_rv = Mock()
        if release_state:
            mock_rv.ok = True
            mock_rv.json.return_value = {
                'state': release_state,
            }
        else:
            mock_rv.status_code = 404
        mock_get.return_value = mock_rv

        with patch('sys.argv', ['fedpkg', '--release', branch, 'retire', 'retire_message']):
            cli = self.new_cli(cfg='fedpkg-test.conf')
            # retire method in rpkg would be called, but there is not environment configured
            # therefore just Exception is caught
            with self.assertRaises(Exception):
                cli.args.path = '/repo_path'
                cli.retire()

    @patch('fedpkg.Commands.is_retired')
    @patch('requests.get')
    def do_not_retire_release(self, branch, release_state, mock_get, is_retired):
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {
            'state': release_state,
        }
        mock_get.return_value = mock_rv
        is_retired.return_value = False

        with patch('sys.argv', ['fedpkg', '--release', branch, 'retire', 'retire_message']):
            cli = self.new_cli(cfg='fedpkg-test.conf')
            # retire is terminated after check
            cli.args.path = '/repo_path'
            cli.retire()

    def test_retire_fedora_release(self):
        self.do_not_retire_release("f30", "disabled")
        self.retire_release("f30", "pending")
        self.do_not_retire_release("f30", "frozen")
        self.do_not_retire_release("f30", "current")
        self.do_not_retire_release("f30", "archived")
        self.retire_release("unknown_fedora_release", None)
        # epel release
        self.retire_release("el6", "disabled")
        self.retire_release("el6", "pending")
        self.retire_release("el6", "frozen")
        self.retire_release("el6", None)
        self.retire_release("epel7", "current")
        self.retire_release("epel7", "archived")
        self.retire_release("epel7", "pending")
        self.retire_release("epel7", None)
        self.retire_release("epel8", None)


class TestSetToken(CliTestCase):
    """
    Test the set-x-token cli command. Pagure / Distgit have the same tests.
    """

    def get_cli(self, cli_cmd, name='fedpkg', cfg=None):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg)

    # Test: using lowercase characters rather than upper-case.
    @patch('getpass.getpass')
    def test_token_input_mixed_lowercase_numerical(self, mock_getpass):

        LOWERCASE_TOKEN = "".join(["x" for _ in range(64)])
        NUMERICAL_TOKEN = "".join([str(i % 10) for i in range(64)])

        MIXED_LOWERCASE_NUMERICAL_TOKEN = ""
        for i in range(32):
            MIXED_LOWERCASE_NUMERICAL_TOKEN += LOWERCASE_TOKEN[i]
            MIXED_LOWERCASE_NUMERICAL_TOKEN += NUMERICAL_TOKEN[i]
        mock_getpass.return_value = MIXED_LOWERCASE_NUMERICAL_TOKEN

        cli_cmd = ['fedpkg', 'set-pagure-token']
        cli = self.get_cli(cli_cmd)

        try:
            cli.set_pagure_token()
        except rpkgError as error:
            expected_error = "ERROR: Token is not properly formatted."
            self.assertEqual(error, expected_error)

    # Test: no input, none input.
    @patch('getpass.getpass')
    def test_token_input_none(self, mock_getpass):

        cli_cmd = ["fedpkg", "set-pagure-token"]
        cli = self.get_cli(cli_cmd)
        mock_getpass.return_value = None

        try:
            cli.set_pagure_token()
        except rpkgError as error:
            expected_error = "ERROR: No input."
            self.assertEqual(error, expected_error)

        """
        EXAMPLE:
            cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path,
                    '--name', 'nethack', 'request-branch', 'f27',
                    '--all-releases']
            cli = self.get_cli(cli_cmd)
            expected_error = \
                'You cannot specify a branch with the "--all-releases" option'
            try:
                cli.request_branch()
                assert False, 'rpkgError not raised'
            except rpkgError as error:
                self.assertEqual(str(error), expected_error)
        """

    # Test: input is only lowercase.
    @patch('getpass.getpass')
    def test_token_input_lowercase(self, mock_getpass):

        LOWERCASE_TOKEN = "".join(["x" for _ in range(64)])

        cli_cmd = ['fedpkg', 'set-pagure-token']
        cli = self.get_cli(cli_cmd)
        mock_getpass.return_value = LOWERCASE_TOKEN

        try:
            cli.set_pagure_token()
        except rpkgError as error:
            expected_error = "ERROR: Token is not properly formatted."
            self.assertEqual(error, expected_error)

    @patch('getpass.getpass')
    def test_token_input_too_short(self, mock_getpass):

        SHORT_TOKEN = "".join(['x' for _ in range(63)])

        cli_cmd = ['fedpkg', 'set-pagure-token']
        cli = self.get_cli(cli_cmd)
        mock_getpass.return_value = SHORT_TOKEN

        try:
            cli.set_pagure_token()
        except rpkgError as error:
            expected_error = "ERROR: Token is not properly formatted."
            self.assertEqual(error, expected_error)

    @patch('getpass.getpass')
    def test_token_input_too_long(self, mock_getpass):

        LONG_TOKEN = "".join(['x' for _ in range(65)])

        cli_cmd = ['fedpkg', 'set-pagure-token']
        cli = self.get_cli(cli_cmd)
        mock_getpass.return_value = LONG_TOKEN

        try:
            cli.set_pagure_token()
        except rpkgError as error:
            expected_error = "ERROR: Token is not properly formatted."
            self.assertEqual(error, expected_error)

    @patch('getpass.getpass')
    def test_token_readable_only_by_user(self, mock_getpass):

        TOKEN = "".join(['X' for _ in range(64)])

        cli_cmd = ['fedpkg', 'set-pagure-token']
        cli = self.get_cli(cli_cmd)
        mock_getpass.return_value = TOKEN

        old_umask = os.umask(0)
        cli.set_pagure_token()
        os.umask(old_umask)

        PATH = os.path.join(os.path.expanduser('~'),
                            '.config',
                            'rpkg',
                            '{0}.conf'.format(cli.name))
        self.assertEqual(os.stat(PATH).st_mode & 0o077, 0)


class TestRequestUnretirement(CliTestCase):
    """Test the request-unretirement cli command."""

    def setUp(self):
        super(TestRequestUnretirement, self).setUp()

    def tearDown(self):
        super(TestRequestUnretirement, self).tearDown()

    def get_cli(self, cli_cmd, name='fedpkg-stage', cfg="fedpkg-stage.conf",
                user_cfg="fedpkg-user.conf"):
        with patch('sys.argv', new=cli_cmd):
            return self.new_cli(name=name, cfg=cfg, user_cfg=user_cfg)

    @patch("fedpkg.cli.get_last_commit_date")
    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement(self, mock_request_post, mock_grb, mock_gcd):
        """Tests request-unretirement command."""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        commit_time = (datetime.now() - timedelta(days=1)).timestamp()
        mock_gcd.return_value = commit_time

        self.run_cmd(['git', 'checkout', '-b', 'f40'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement']

        cli = self.get_cli(cli_cmd)
        cli.request_unretirement()

        expected_issue_content = {
            'action': 'unretirement',
            'name': 'testpkg',
            'type': 'rpms',
            'branches': ['f40'],
            'review_bugzilla': None,
            'maintainer': 'root',
        }

        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/41')
        self.assertEqual(output, expected_output)

    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement_not_release_branch(self, mock_request_post, mock_grb):
        """Tests request-unretirement command on branch that is not a release branch."""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        expected_error = (
            "No requested branches {0} are release branches. So there is nothing to unretire."
            .format(['f15']))

        self.run_cmd(['git', 'checkout', '-b', 'f15'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement']

        cli = self.get_cli(cli_cmd)
        try:
            cli.request_unretirement()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch("fedpkg.cli.get_last_commit_date")
    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement_bugzilla_needed_but_not_provided(
            self, mock_request_post, mock_grb, mock_gcd
    ):
        """Tests request-unretirement that require to check bugzilla, but bug id wasn't provided"""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        commit_time = (datetime.now() - timedelta(days=57)).timestamp()
        mock_gcd.return_value = commit_time

        expected_error = (
            "Bugzilla url should be provided, "
            "because last commit was made more than 8 weeks ago."
        )

        self.run_cmd(['git', 'checkout', '-b', 'f40'], cwd=self.cloned_repo_path)

        cli_cmd = ['fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement']
        cli = self.get_cli(cli_cmd)

        try:
            cli.request_unretirement()
            assert False, 'rpkgError not raised'
        except rpkgError as error:
            self.assertEqual(str(error), expected_error)

    @patch("fedpkg.bugzilla.BugzillaClient.get_review_bug")
    @patch("fedpkg.cli.get_last_commit_date")
    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement_bz_id_provided(
            self, mock_request_post, mock_grb, mock_gcd, mock_bz_grb
    ):
        """Tests request-unretirement command when bz_id provided"""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        commit_time = (datetime.now() - timedelta(days=57)).timestamp()
        mock_gcd.return_value = commit_time

        self.run_cmd(['git', 'checkout', '-b', 'f40'], cwd=self.cloned_repo_path)

        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement',
            '--bz_bug_id', '1234567']

        cli = self.get_cli(cli_cmd)
        cli.request_unretirement()

        expected_issue_content = {
            'action': 'unretirement',
            'name': 'testpkg',
            'type': 'rpms',
            'branches': ['f40'],
            'review_bugzilla': '1234567',
            'maintainer': 'root',
        }

        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/41')
        self.assertEqual(output, expected_output)

    @patch("fedpkg.bugzilla.BugzillaClient.get_review_bug")
    @patch("fedpkg.cli.get_last_commit_date")
    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement_bz_url_provided(
            self, mock_request_post, mock_grb, mock_gcd, mock_bz_grb
    ):
        """Tests request-unretirement command when bz url provided"""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        commit_time = (datetime.now() - timedelta(days=57)).timestamp()
        mock_gcd.return_value = commit_time

        self.run_cmd(['git', 'checkout', '-b', 'f40'], cwd=self.cloned_repo_path)

        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement',
            '--bz_bug_id', 'https://bugzilla.redhat.com/show_bug.cgi?id=1234567']

        cli = self.get_cli(cli_cmd)
        cli.request_unretirement()

        expected_issue_content = {
            'action': 'unretirement',
            'name': 'testpkg',
            'type': 'rpms',
            'branches': ['f40'],
            'review_bugzilla': '1234567',
            'maintainer': 'root',
        }

        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/41')
        self.assertEqual(output, expected_output)

    @patch("fedpkg.cli.get_last_commit_date")
    @patch("fedpkg.cli.get_release_branches")
    @patch("requests.post")
    @patch('sys.stdout', new=io.StringIO())
    def test_request_unretirement_repo_name_provided(
            self, mock_request_post, mock_grb, mock_gcd
    ):
        """Tests request-unretirement command."""
        mock_grb.return_value = {
            'fedora': ['f39', 'f40', 'f41'],
            'epel': ['epel9', 'epel10'],
        }

        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'issue': {'id': 41}}
        mock_request_post.return_value = mock_rv

        commit_time = (datetime.now() - timedelta(days=1)).timestamp()
        mock_gcd.return_value = commit_time

        self.run_cmd(['git', 'checkout', '-b', 'f40'], cwd=self.cloned_repo_path)
        cli_cmd = [
            'fedpkg-stage', '--path', self.cloned_repo_path, 'request-unretirement',
            '--repo', 'belusha'
        ]
        cli = self.get_cli(cli_cmd)
        cli.request_unretirement()

        expected_issue_content = {
            'action': 'unretirement',
            'name': 'belusha',
            'type': 'rpms',
            'branches': ['f40'],
            'review_bugzilla': None,
            'maintainer': 'root',
        }

        post_data = mock_request_post.call_args_list[0][1]['data']
        actual_issue_content = json.loads(json.loads(
            post_data)['issue_content'].strip('```'))
        self.assertEqual(expected_issue_content, actual_issue_content)
        output = sys.stdout.getvalue().strip()
        expected_output = ('https://pagure.stg.example.com/releng/'
                           'fedora-scm-requests/issue/41')
        self.assertEqual(output, expected_output)
