# -*- coding: utf-8 -*-

import configparser
import os
import shutil
import subprocess
import tempfile
from unittest import mock, TestCase

from pyrpkg.errors import rpkgError

from fedpkg.cli import fedpkgClient

TEST_CONFIG = os.path.join(os.path.dirname(__file__), 'fedpkg-test.conf')


class RetireTestCase(TestCase):
    def setUp(self):
        self.tmpdir = tempfile.mkdtemp()
        self.log = mock.Mock()

    def tearDown(self):
        shutil.rmtree(self.tmpdir)

    def _setup_repo(self, origin):
        subprocess.check_call(
            ['git', 'init'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['git', 'config', 'user.name', 'John Doe'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['git', 'config', 'user.email', 'jdoe@example.com'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['git', 'remote', 'add', 'origin', origin],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['touch', 'fedpkg.spec'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['git', 'add', '.'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        subprocess.check_call(
            ['git', 'commit', '-m', 'Initial commit'],
            cwd=self.tmpdir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    def _get_latest_commit(self):
        proc = subprocess.Popen(['git', 'log', '-n', '1', '--pretty=%s'],
                                cwd=self.tmpdir, stdout=subprocess.PIPE,
                                universal_newlines=True)
        out, err = proc.communicate()
        return out.strip()

    def _fake_client(self, args):
        config = configparser.ConfigParser()
        config.read(TEST_CONFIG)
        with mock.patch('sys.argv', new=args):
            client = fedpkgClient(config)
            client.do_imports(site='fedpkg')
            client.setupLogging(self.log)

            client.parse_cmdline()
            client.args.path = self.tmpdir
            client.cmd.push = mock.Mock()
        return client

    def assertRetired(self, reason):
        self.assertTrue(os.path.isfile(os.path.join(self.tmpdir,
                                                    'dead.package')))
        self.assertFalse(os.path.isfile(os.path.join(self.tmpdir,
                                                     'fedpkg.spec')))
        self.assertEqual(self._get_latest_commit(), reason)

    @mock.patch('requests.post')
    @mock.patch("requests.get", new=lambda *args, **kwargs: mock.Mock(status_code=404))
    def test_retire_with_namespace(self, requests_post):
        mock_rv = mock.Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {"anitya_status": "no-monitoring"}
        requests_post.return_value = mock_rv

        self._setup_repo('ssh://git@pkgs.example.com/rpms/fedpkg')
        args = ['fedpkg', '--release=rawhide', 'retire', 'my reason']

        client = self._fake_client(args)
        client.retire()

        self.assertRetired('my reason')
        self.assertEqual(len(client.cmd.push.call_args_list), 1)

        args = ['fedpkg', '--release=rawhide', 'disable-monitoring']
        client = self._fake_client(args)
        client.do_disable_monitoring()

        url = 'https://src.example.com/_dg/anitya/rpms/fedpkg'
        data = '{"anitya_status": "no-monitoring"}'
        headers = {
            'Authorization': 'token notsecretatall',
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
        requests_post.assert_called_once_with(url, data=data, headers=headers, timeout=90)

    @mock.patch('requests.post')
    @mock.patch("requests.get", new=lambda *args, **kwargs: mock.Mock(status_code=404))
    def test_retire_with_namespace_disable_monitoring_fails(self, requests_post):
        """
        retire operation itself runs fine, but subsequent procedure disable-monitoring fails
        """
        mock_rv = mock.Mock()
        mock_rv.ok = False
        mock_rv.json.return_value = {"error": "Invalid or expired token"}
        requests_post.return_value = mock_rv

        self._setup_repo('ssh://git@pkgs.example.com/rpms/fedpkg')
        args = ['fedpkg', '--release=rawhide', 'retire', 'my reason']

        client = self._fake_client(args)
        client.retire()

        self.assertRetired('my reason')
        self.assertEqual(len(client.cmd.push.call_args_list), 1)

        args = ['fedpkg', '--release=rawhide', 'disable-monitoring']
        client = self._fake_client(args)
        self.assertRaisesRegex(
            rpkgError,
            "The following error occurred while disabling monitoring: Invalid or expired token\n"
            "For invalid or expired tokens please set a new token",
            client.do_disable_monitoring
        )

        url = 'https://src.example.com/_dg/anitya/rpms/fedpkg'
        data = '{"anitya_status": "no-monitoring"}'
        headers = {
            'Authorization': 'token notsecretatall',
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
        requests_post.assert_called_once_with(url, data=data, headers=headers, timeout=90)

    @mock.patch("requests.get", new=lambda *args, **kwargs: mock.Mock(status_code=404))
    def test_retire_without_namespace(self):
        self._setup_repo('ssh://git@pkgs.example.com/fedpkg')
        args = ['fedpkg', '--release=rawhide', 'retire', 'my reason']

        client = self._fake_client(args)
        client.retire()

        self.assertRetired('my reason')
        self.assertEqual(len(client.cmd.push.call_args_list), 1)

    @mock.patch("requests.get", new=lambda *args, **kwargs: mock.Mock(status_code=404))
    def test_package_is_retired_already(self):
        self._setup_repo('ssh://git@pkgs.example.com/fedpkg')
        with open(os.path.join(self.tmpdir, 'dead.package'), 'w') as f:
            f.write('dead package')

        args = ['fedpkg', '--release=rawhide', 'retire', 'my reason']
        client = self._fake_client(args)
        client.log = mock.Mock()
        client.retire()
        args, kwargs = client.log.warning.call_args
        self.assertIn('dead.package found, package or module is already retired',
                      args[0])

    @mock.patch(
        "requests.get",
        new=lambda *args, **kwargs: mock.Mock(
            status_code=200, ok=True, json=lambda: {"state": "archived"}
        ),
    )
    def test_package_on_retired(self):
        self._setup_repo("ssh://git@pkgs.example.com/fedpkg")
        args = ["fedpkg", "--release=rawhide", "retire", "my reason"]

        client = self._fake_client(args)
        client.retire()
        args, kwargs = client.log.error.call_args
        self.assertIn("retire operation is not allowed", args[0])
