# -*- coding: utf-8 -*-
# fedpkg - a Python library for RPM Packagers
#
# Copyright (C) 2017 Red Hat Inc.
# Author(s): Matt Prahl <mprahl@redhat.com>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
# the full text of the license.

from configparser import NoOptionError, NoSectionError
import json
import unittest
from unittest.mock import Mock, patch

import git
from requests.exceptions import ConnectionError

from fedpkg import utils
from freezegun import freeze_time
from pyrpkg.errors import rpkgError
import requests


class TestUtils(unittest.TestCase):
    """Test functions in fedpkg.utils"""

    def test_verify_sls(self):
        """Test verify_sls"""
        sls = {"security_fixes": "2222-12-01"}
        # If it's invalid, an rpkgError will be raised
        try:
            utils.verify_sls(sls)
        except rpkgError:
            assert False, 'An rpkgError exception was raised but not expected'

    def test_verify_sls_eol_expired(self):
        """Test verify_sls raises an exception when an EOL is expired"""
        sls = {"security_fixes": "2001-12-01"}

        try:
            utils.verify_sls(sls)
            assert False, "An rpkgError exception was not raised"
        except rpkgError as e:
            self.assertEqual(str(e), 'The SL "2001-12-01" is already expired')

    def test_sl_list_dict(self):
        """Test sl_list_to_dict"""
        sls = ['security_fixes:2030-01-01', 'bug_fixes:2029-01-01']
        sls_dict = {'security_fixes': '2030-01-01', 'bug_fixes': '2029-01-01'}
        self.assertEqual(utils.sl_list_to_dict(sls), sls_dict)

    def test_sl_list_to_dict_invalid_format(self):
        """Tests sl_list_to_dict with an invalid SL format. An error is
        expected.
        """
        try:
            sls = ['security_fixes:2030-12-01', 'bug_fixes/2030-12-01']
            utils.sl_list_to_dict(sls)
            assert False, 'An rpkgError exception was not raised'
        except rpkgError as e:
            assert str(e) == \
                'The SL "bug_fixes/2030-12-01" is in an invalid format'

    def test_verify_sls_invalid_date(self):
        """Test verify_sls with an SL that is not June 1st or December 1st. An
        error is expected.
        """
        for eol in ['2030-01-01', '2030-12-25']:
            try:
                sls = {"security_fixes": eol, "bug_fixes": eol}
                utils.verify_sls(sls)
                assert False, "An rpkgError exception was not raised"
            except rpkgError as e:
                assert str(e) == ('The SL "{0}" must expire on June 1st or '
                                  'December 1st'.format(eol))

    @patch('requests.get')
    def test_get_release_branches(self, mock_request_get):
        """Test that get_release_branches returns all the active Fedora release
        branches.
        """
        mock_rv = Mock()
        mock_rv.ok = True
        # This abbreviated data returned from the Bodhi API active releases
        mock_rv.json.return_value = {'releases': [
            {'name': 'EPEL-7', 'branch': 'epel7'},
            {'name': 'EPEL-8', 'branch': 'epel8'},
            {'name': 'EPEL-8N',
             'long_name': 'Fedora EPEL 8 Next',
             'version': '8',
             'id_prefix': 'FEDORA-EPEL-NEXT',
             'branch': 'epel8-next'},
            {'name': 'EPEL-9', 'branch': 'epel9'},
            {'name': 'EPEL-9N', 'branch': 'epel9-next'},
            {'name': 'F38', 'branch': 'f38'},
            {'name': 'F38C', 'branch': 'f38'},
            {'name': 'F38F', 'branch': 'f38'},
            {'name': 'F38M', 'branch': 'f38m'},
            {'name': 'F39', 'branch': 'f39'},
            {'name': 'F39C', 'branch': 'f39'},
            {'name': 'F39F', 'branch': 'f39'},
            {'name': 'F40', 'branch': 'f40'},
            {'name': 'F40C', 'branch': 'f40'},
            {'name': 'F40F', 'branch': 'f40'},
            {'name': 'F41', 'branch': 'f41'},
            {'name': 'F41C', 'branch': 'f41'},
            {'name': 'F41F', 'branch': 'f41'},
            {'name': 'F42', 'branch': 'f42'},
            ], 'page': 1, 'pages': 1, 'rows_per_page': 20, 'total': 12}
        mock_request_get.return_value = mock_rv
        expected = {
            'epel': ['epel7', 'epel8', 'epel8-next', 'epel9', 'epel9-next'],
            'fedora': ['f38', 'f38m', 'f39', 'f40', 'f41'],
        }
        actual = utils.get_release_branches('http://src.local')
        actual_sorted = {key: sorted(value) for key, value in sorted(actual.items())}
        self.assertDictEqual(expected, actual_sorted)


@patch('requests.get')
class TestAssertNewTestsRepo(unittest.TestCase):
    """Test assert_new_tests_repo"""

    def test_should_raise_error_if_connection_error_to_distgit(self, get):
        get.side_effect = ConnectionError

        self.assertRaisesRegex(
            rpkgError, 'The error was',
            utils.assert_new_tests_repo, 'testrepo', 'http://distgit/')

    def test_test_repo_exists(self, get):
        get.return_value = Mock(ok=True)

        self.assertRaisesRegex(
            rpkgError, 'Repository .+ already exists',
            utils.assert_new_tests_repo, 'testrepo', 'http://distgit/')

    def test_keep_quiet_if_repo_not_exist(self, get):
        get.return_value = Mock(ok=False)
        utils.assert_new_tests_repo('testrepo', 'http://distgit/')


class TestGetPagureToken(unittest.TestCase):
    """Test obtaining of Pagure token"""

    def test_return_token(self):
        config = Mock()
        config.get.return_value = '123456'

        token = utils.config_get_safely(config, 'fedpkg.pagure', 'token')

        self.assertEqual('123456', token)
        config.get.assert_called_once_with('fedpkg.pagure', 'token')

    def test_config_does_not_have_token(self):
        config = Mock()

        config.get.side_effect = NoOptionError('token', 'fedpkg.pagure')
        self.assertRaisesRegex(rpkgError, "Missing option 'token' in the section 'fedpkg.pagure'",
                               utils.config_get_safely,
                               config,
                               'fedpkg.pagure', 'token')

        config.get.side_effect = NoSectionError('fedpkg.pagure')
        self.assertRaisesRegex(rpkgError, "Missing section 'fedpkg.pagure'",
                               utils.config_get_safely, config, 'fedpkg.pagure', 'token')


class TestVerifySLS(unittest.TestCase):
    """Test verify_sls"""

    def test_sl_date_format_is_invalid(self):
        self.assertRaisesRegex(
            rpkgError,
            "The EOL date .+ is in an invalid format",
            utils.verify_sls,
            {"bug_fixes": "2018/7/21"},
        )

    @freeze_time("2018-01-01")
    @patch("requests.get")
    def test_keep_quiet_if_service_levels_are_ok(self, get):
        rv = Mock(ok=True)
        rv.json.side_effect = [
            {
                'count': 1,
                'results': [{
                    'id': 1,
                    'name': 'bug_fixes',
                    'description': 'Bug fixes'
                }],
            },
            {
                'count': 1,
                'results': [{
                    'id': 2,
                    'name': 'security_fixes',
                    'description': 'Security fixes'
                }],
            }
        ]
        get.return_value = rv

        utils.verify_sls(
            {"bug_fixes": "2018-06-01", "security_fixes": "2018-12-01"},
        )


class TestIsEpel(unittest.TestCase):
    """Test is_epel"""

    def test_valid_epel_branch_names(self):
        self.assertEqual(utils.is_epel("el6"), True)
        self.assertEqual(utils.is_epel("epel7"), True)
        self.assertEqual(utils.is_epel("epel8"), True)
        self.assertEqual(utils.is_epel("epel8-next"), True)
        self.assertEqual(utils.is_epel("epel9"), True)
        self.assertEqual(utils.is_epel("epel9-next"), True)
        self.assertEqual(utils.is_epel("epel10"), True)
        self.assertEqual(utils.is_epel("epel10.1"), True)
        self.assertEqual(utils.is_epel("epel10.10"), True)
        self.assertEqual(utils.is_epel("epel10.11"), True)

        self.assertEqual(utils.is_epel("f36"), False)
        self.assertEqual(utils.is_epel("rawhide"), False)


@patch('requests.get')
class TestAssertValidEPELPackage(unittest.TestCase):
    """Test assert_valid_epel_package"""

    def test_raise_error_if_connection_error(self, get):
        get.side_effect = ConnectionError

        self.assertRaisesRegex(
            rpkgError, 'The error was:',
            utils.assert_valid_epel_package, 'pkg', 'epel7')

    def test_raise_error_if_GET_response_not_ok(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel7')

    def test_correct_url_epel7(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel7')
        get.assert_called_once_with(
            'https://infrastructure.fedoraproject.org/repo/json/pkg_el7.json',
            timeout=60)

    def test_correct_url_epel8(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel8')
        get.assert_called_once_with(
            'https://infrastructure.fedoraproject.org/repo/json/pkg_el8.json',
            timeout=60)

    def test_correct_url_epel8_next(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel8-next')
        get.assert_called_once_with(
            'https://infrastructure.fedoraproject.org/repo/json/pkg_el8.json',
            timeout=60)

    def test_correct_url_epel9(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel9')
        get.assert_called_once_with(
            'https://composes.stream.centos.org/production/'
            'latest-CentOS-Stream/compose/metadata/rpms.json',
            timeout=60)

    def test_correct_url_epel10(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel10')
        get.assert_called_once_with(
            'https://composes.stream.centos.org/stream-10/production'
            '/latest-CentOS-Stream/compose/metadata/rpms.json',
            timeout=60)

    def test_correct_url_epel10_10(self, get):
        get.return_value = Mock(ok=False, status_code=404)

        self.assertRaisesRegex(
            rpkgError, 'The status code was: 404',
            utils.assert_valid_epel_package, 'pkg', 'epel10.10')
        get.assert_called_once_with(
            'https://composes.stream.centos.org/stream-10/production'
            '/latest-CentOS-Stream/compose/metadata/rpms.json',
            timeout=60)

    def test_should_not_have_epel_branch_for_el6_pkg(self, get):
        get.return_value.json.return_value = {
            'arches': [
                'i686', 'noarch', 'i386', 'ppc64', 'ppc', 'x86_64'
            ],
            'packages': {
                'pkg1': {
                    # For el6, these arches will cause error raised.
                    'arch': ['i686', 'noarch', 'ppc64', 'x86_64']
                }
            }
        }

        self.assertRaisesRegex(
            rpkgError, 'is built on all supported arches',
            utils.assert_valid_epel_package, 'pkg1', 'el6')

    def test_should_not_have_epel_branch_for_el7_pkg(self, get):
        get.return_value.json.return_value = {
            'arches': [
                'i686', 'noarch', 'i386', 'ppc64', 'ppc', 'x86_64'
            ],
            'packages': {
                'pkg1': {
                    # For epel7, these arches will cause error raised.
                    'arch': ['i386', 'noarch', 'ppc64', 'x86_64']
                }
            }
        }

        self.assertRaisesRegex(
            rpkgError, 'is built on all supported arches',
            utils.assert_valid_epel_package, 'pkg1', 'epel7')

    def test_raise_error_if_package_has_noarch_only(self, get):
        get.return_value.json.return_value = {
            'arches': [
                'i686', 'noarch', 'i386', 'ppc64', 'ppc', 'x86_64'
            ],
            'packages': {
                'pkg1': {
                    'arch': ['noarch']
                }
            }
        }

        self.assertRaisesRegex(
            rpkgError, 'This package is already an EL package',
            utils.assert_valid_epel_package, 'pkg1', 'epel7')


@patch('requests.post')
class TestNewPagureIssue(unittest.TestCase):
    """Test new_pagure_issue"""

    def test_raise_error_if_connection_error(self, post):
        post.side_effect = ConnectionError
        logger = Mock()

        self.assertRaisesRegex(
            rpkgError, 'The connection to Pagure failed',
            utils.new_pagure_issue,
            logger, 'http://distgit/', '123456', 'new package', {'repo': 'pkg1'}, 'fedpkg')

    def test_responses_not_ok_and_response_body_is_not_json(self, post):
        rv = Mock(ok=False, text='error')
        rv.json.side_effect = ValueError
        post.return_value = rv
        logger = Mock()

        self.assertRaisesRegex(
            rpkgError,
            'The following error occurred while creating a new issue',
            utils.new_pagure_issue,
            logger, 'http://distgit/', '123456', 'new package', {'repo': 'pkg1'}, 'fedpkg')

    def test_responses_not_ok_when_token_is_expired(self, post):
        rv = Mock(
            ok=False,
            text='Invalid or expired token. Please visit '
                 'https://pagure.io/settings#api-keys to get or renew your API token.')
        rv.json.side_effect = ValueError
        post.return_value = rv
        logger = Mock()

        self.assertRaisesRegex(
            rpkgError,
            '\nFor invalid or expired tokens please '
            'set a new token in your user configuration by running:'
            '\n\n\tfedpkg set-pagure-token\n\n'
            'The command is interactive; enter the new token when prompted.',
            utils.new_pagure_issue,
            logger, 'http://distgit/', '123456', 'new package', {'repo': 'pkg1'}, 'fedpkg')

    def test_create_pagure_issue(self, post):
        rv = Mock(ok=True)
        rv.json.return_value = {'issue': {'id': 1}}
        post.return_value = rv
        logger = Mock()

        pagure_api_url = 'http://distgit'
        issue_ticket_body = {'repo': 'pkg1'}

        issue_url = utils.new_pagure_issue(logger,
                                           pagure_api_url,
                                           '123456',
                                           'new package',
                                           issue_ticket_body,
                                           'fedpkg',)

        expected_issue_url = (
            '{0}/releng/fedora-scm-requests/issue/1'
            .format(pagure_api_url)
        )
        self.assertEqual(expected_issue_url, issue_url)

        post.assert_called_once_with(
            '{0}/api/0/releng/fedora-scm-requests/new_issue'
            .format(pagure_api_url),
            headers={
                'Authorization': 'token {0}'.format(123456),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            },
            data=json.dumps({
                'title': 'new package',
                'issue_content': issue_ticket_body,
            }),
            timeout=60
        )


@patch("requests.get")
class TestQueryBodhi(unittest.TestCase):
    """Test suite for utils.query_bodhi"""

    def test_connection_error(self, mock_get):
        """Test that a ConnectionError raises an rpkgError with the correct message."""
        # Simulate a connection error when making the get request
        mock_get.side_effect = requests.exceptions.ConnectionError('Mocked connection error')

        with self.assertRaises(rpkgError) as cm:
            utils.query_bodhi('http://localhost/')

        self.assertIn(
            'The connection to BODHI failed while trying to get the active release branches. '
            'The error was: Mocked connection error',
            str(cm.exception),
            "Expected error message is missing or incorrect"
        )

    def test_response_not_ok(self, mock_get):
        """Test that a non-OK response raises an rpkgError with the correct message."""
        # Mock a response with .ok as False and a text message
        mock_rv = Mock()
        mock_rv.ok = False
        mock_rv.text = 'Mocked error message'
        mock_get.return_value = mock_rv

        with self.assertRaises(rpkgError) as cm:
            utils.query_bodhi('http://localhost/')

        self.assertIn(
            'The following error occurred while trying to get the active release '
            'branches in Bodhi: Mocked error message',
            str(cm.exception),
            "Expected error message is missing or incorrect"
        )

    def test_read_data_normally(self, mock_get):
        """Test that query_bodhi returns the correct branch list from a normal response."""
        # Mock a valid response from the Bodhi API
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {
            'releases': [
                {'name': 'F40', 'branch': 'f40'},
                {'name': 'F41', 'branch': 'f41'},
                {'name': 'Rawhide Release', 'branch': 'rawhide', 'version': '42'},
                {'name': 'EPEL9', 'branch': 'epel9'},
                {'name': 'EPEL8', 'branch': 'epel8'}
            ]
        }
        mock_get.return_value = mock_rv

        # The expected branch list to be returned by query_bodhi
        expected = ['f40', 'f41', 'f42', 'epel9', 'epel8']

        # Call query_bodhi and compare the result to the expected list
        result = utils.query_bodhi('http://localhost/')
        self.assertEqual(result, expected, f"Expected branches {expected}, but got {result}")

    def test_rawhide_not_present(self, mock_get):
        """Test that if rawhide is not present in the releases,
        it does not construct a rawhide branch."""
        # Mock a response without a rawhide entry
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {
            'releases': [
                {'name': 'F40', 'branch': 'f40'},
                {'name': 'F41', 'branch': 'f41'},
                {'name': 'EPEL9', 'branch': 'epel9'},
                {'name': 'EPEL8', 'branch': 'epel8'}
            ]
        }
        mock_get.return_value = mock_rv

        # The expected branch list without a rawhide branch
        expected = ['f40', 'f41', 'epel9', 'epel8']

        # Call query_bodhi and compare the result to the expected list
        result = utils.query_bodhi('http://localhost/')
        self.assertEqual(result, expected, f"Expected branches {expected}, but got {result}")

    def test_empty_releases(self, mock_get):
        """Test that an empty releases list returns an empty branches list."""
        # Mock a response with no releases
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {'releases': []}
        mock_get.return_value = mock_rv

        # Expected branches should be an empty list
        expected = []

        # Call query_bodhi and compare the result to the expected empty list
        result = utils.query_bodhi('http://localhost/')
        self.assertEqual(result, expected, "Expected an empty list of branches, but got some")


class TestGetStreamBranches(unittest.TestCase):
    """Test get_stream_branches"""

    @patch('fedpkg.utils.get_pagure_branches')
    @patch('requests.get')
    def test_fedora_and_epel_branches_are_filtered_out(self, get, pagure_branches):
        logger = Mock()
        apibaseurl = "https://bodhiurl"
        rv = Mock(ok=True)
        rv.json.return_value = {
            'releases': [
                {'name': 'ELN', 'branch': 'eln', 'version': '8'},
                {'name': 'F40', 'branch': 'rawhide', 'version': '40'},
                {'name': 'F40C', 'branch': 'f40', 'version': '40'},
                {'name': 'epel8', 'branch': 'epel8', 'version': '8'},
            ],
            'page': 1,
            'pages': 1,
            'rows_per_page': 20,
            'total': 4
        }
        get.return_value = rv
        pagure_branches.return_value = ["epel7", "epel8", "epel9", "f38", "f39"]

        result = utils.get_stream_branches('http://localhost/', 'pkg', apibaseurl, logger)
        self.assertEqual(['epel8'], list(result))


class TestExpandRelease(unittest.TestCase):
    """Test expand_release"""

    def setUp(self):
        self.releases = {
            'fedora': ['f28', 'f27'],
            'epel': ['el6', 'epel7']
        }

    def test_expand_fedora(self):
        result = utils.expand_release('fedora', self.releases)
        self.assertEqual(self.releases['fedora'], result)

    def test_expand_epel(self):
        result = utils.expand_release('epel', self.releases)
        self.assertEqual(self.releases['epel'], result)

    def test_expand_main(self):
        result = utils.expand_release('main', self.releases)
        self.assertEqual(['rawhide'], result)

    def test_expand_rawhide(self):
        result = utils.expand_release('rawhide', self.releases)
        self.assertEqual(['rawhide'], result)

    def test_normal_release(self):
        result = utils.expand_release('f28', self.releases)
        self.assertEqual(['f28'], result)

        result = utils.expand_release('el6', self.releases)
        self.assertEqual(['el6'], result)

    def test_expand_unknown_name(self):
        result = utils.expand_release('some_branch', self.releases)
        self.assertEqual(None, result)


class TestGetFedoraReleaseState(unittest.TestCase):

    @patch('requests.get')
    def test_release_exists_current(self, mock_get):
        mock_rv = Mock()
        mock_rv.ok = True
        mock_rv.json.return_value = {
            'name': 'F30',
            'long_name': 'Fedora 30',
            'version': '30',
            'branch': 'f30',
            'state': 'current',  # this item is only important for the test
        }
        mock_get.return_value = mock_rv

        config = Mock()
        config.get.return_value = 'https://service_url/releases/F30'

        rv = utils.get_fedora_release_state(config, 'fedpkg', 'F30')
        self.assertEqual(rv, 'current')

    @patch('requests.get')
    def test_release_does_not_exist(self, mock_get):
        mock_rv = Mock()
        mock_rv.ok = False
        mock_rv.status_code = 404
        mock_get.return_value = mock_rv

        config = Mock()
        config.get.return_value = 'https://service_url/releases/eng-fedora-29'

        rv = utils.get_fedora_release_state(config, 'fedpkg', 'eng-fedora-29')
        self.assertEqual(rv, None)

    def test_config_does_not_have_option(self):
        config = Mock()
        config.get.side_effect = NoOptionError('releases_service', 'fedpkg.bodhi')
        self.assertRaisesRegex(rpkgError, r"Could not get release state for Fedora \(F30M\): "
                               "No option 'releases_service' in section: 'fedpkg.bodhi'.",
                               utils.get_fedora_release_state, config, 'fedpkg', 'F30M')


class TestGetLastCommitDate(unittest.TestCase):
    """Test get_last_commit_date"""

    @patch("git.Repo")
    def test_get_last_commit_date_success(self, mock_git_repo):
        expected_date = 1712761897
        mock_repo = mock_git_repo.clone_from.return_value
        mock_commit = mock_repo.head.commit
        mock_commit.committed_date = expected_date

        result = utils.get_last_commit_date('base_url', 'namespace',
                                            'repo_name', 'rawhide')

        self.assertEqual(result, expected_date)

    @patch("git.Repo")
    def test_get_last_commit_date_exception(self, mock_git_repo):
        error_msg = ("Unable to get last commit date. Try to check repo name and namespace "
                     "if it exists.")
        mock_git_repo.clone_from.side_effect = git.exc.GitCommandError("some error")

        with self.assertRaises(rpkgError, msg=error_msg):
            utils.get_last_commit_date(
                "url", "ns", "name", "branch")


class TestGetReleaseBranches(unittest.TestCase):
    """Test utils.get_release_branches"""

    @patch('fedpkg.utils.query_bodhi')
    def test_get_release_branches_excludes_rawhide(self, mock_query_bodhi):
        """Test that get_release_branches correctly excludes the rawhide branch."""
        # Mocking the branches returned by query_bodhi
        mock_query_bodhi.return_value = ['f40', 'f41', 'f42', 'epel9', 'epel8']  # f42 is rawhide

        expected_output = {
            'fedora': ['f40', 'f41'],
            'epel': ['epel8', 'epel9']
        }

        # Run the method and assert that rawhide branch (f42) is excluded
        result = utils.get_release_branches('https://bodhi.fedoraproject.org/releases/')
        self.assertEqual(result, expected_output,
                         msg=f"Expected branches {expected_output}, but got {result}")


class TestDoesPackageExistInAnitya(unittest.TestCase):
    """Test utils.does_package_exist_in_anitya"""

    @patch('requests.get')
    def test_does_package_exist_in_anitya_everything_is_okay(self, mock_get):
        """Test when package exists in Anitya (status code 200)"""
        expected_output = True
        mock_response = Mock()
        mock_response.status_code = 200
        mock_get.return_value = mock_response
        package_name = 'package_name'
        rv = utils.does_package_exist_in_anitya(package_name)
        self.assertEqual(rv, expected_output)

    @patch('requests.get')
    def test_does_package_exist_in_anitya_no_package_exist(self, mock_get):
        """Test when package doesn't exist in Anitya (status code 400)"""
        expected_output = False
        mock_response = Mock()
        mock_response.status_code = 400
        mock_get.return_value = mock_response
        package_name = 'package_name'
        rv = utils.does_package_exist_in_anitya(package_name)
        self.assertEqual(rv, expected_output)

    @patch('requests.get')
    def test_does_package_exist_in_anitya_exception_raises(self, mock_get):
        """Test when there's a request exception"""
        mock_get.side_effect = requests.exceptions.RequestException("RequestException")
        result = utils.does_package_exist_in_anitya("error-package")
        self.assertEqual(result, False)
